<?php $__env->startSection('title', __('Harsh Driving Analysis')); ?>

<?php $__env->startSection('content'); ?>
<div class="page-wrapper" id="harsh-driving-report">
    <div class="page-header d-print-none">
        <div class="container-xl">
            <div class="row g-2 align-items-center">
                <div class="col">
                    <div class="page-pretitle">
                        <a href="/reports" class="text-muted"><?php echo e(__('Reports')); ?></a> / <?php echo e(__('Security and Compliance
                        Reports')); ?>

                    </div>
                    <h2 class="page-title">
                        <i class="ti ti-arrow-autofit-content me-2"></i>
                        <?php echo e(__('Harsh Driving Analysis')); ?>

                    </h2>
                    <div class="text-muted mt-1">
                        <?php echo e(__('Sudden acceleration and deceleration detection')); ?>

                    </div>
                </div>
            </div>
        </div>
    </div>

    <div class="page-body">
        <div class="container-xl">
            <!-- Filters -->
            <div class="card mb-3">
                <div class="card-header">
                    <h3 class="card-title"><?php echo e(__('Filters')); ?></h3>
                </div>
                <div class="card-body">
                    <div class="row g-3">
                        <div class="col-md-3">
                            <label class="form-label"><?php echo e(__('Vehicle')); ?></label>
                            <div class="input-group">
                                <select class="form-select" id="vehicleSelect">
                                    <option value=""><?php echo e(__('Select Vehicle')); ?></option>
                                </select>
                                <button type="button" class="btn btn-outline-secondary d-none" id="vehicleSettingsLink"
                                    onclick="goToVehicleSettings()" title="<?php echo e(__('Go to Vehicle Settings')); ?>">
                                    <i class="icon ti ti-settings"></i>
                                </button>
                            </div>
                        </div>
                        <div class="col-md-3">
                            <label class="form-label"><?php echo e(__('Start Date')); ?></label>
                            <input type="date" class="form-control" id="startDate"
                                value="<?php echo e(date('Y-m-d', strtotime('-1 week'))); ?>">
                        </div>
                        <div class="col-md-3">
                            <label class="form-label"><?php echo e(__('End Date')); ?></label>
                            <input type="date" class="form-control" id="endDate" value="<?php echo e(date('Y-m-d')); ?>">
                        </div>
                        <div class="col-md-2">
                            <label class="form-label"><?php echo e(__('Deceleration/Acceleration Limit (km/s²)')); ?></label>
                            <input type="number" class="form-control" id="accelerationLimit" value="15" min="1" max="20">
                        </div>
                        <div class="col-md-1 d-flex align-items-end">
                            <button type="button" class="btn btn-primary w-100" onclick="generateReport()">
                                <i class="icon ti ti-search me-1"></i>
                                <?php echo e(__('Search')); ?>

                            </button>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Summary Cards -->
            <div class="row row-cards mb-3" id="summaryCards">
                <div class="col-sm-6 col-lg-3">
                    <div class="card">
                        <div class="card-body">
                            <div class="subheader"><?php echo e(__('Total Events')); ?></div>
                            <div class="h1 mb-0" id="totalEvents">-</div>
                        </div>
                    </div>
                </div>
                <div class="col-sm-6 col-lg-3">
                    <div class="card">
                        <div class="card-body">
                            <div class="subheader"><?php echo e(__('Acceleration Events')); ?></div>
                            <div class="h1 mb-0" id="accelerationEvents">-</div>
                        </div>
                    </div>
                </div>
                <div class="col-sm-6 col-lg-3">
                    <div class="card">
                        <div class="card-body">
                            <div class="subheader"><?php echo e(__('Deceleration Events')); ?></div>
                            <div class="h1 mb-0" id="decelerationEvents">-</div>
                        </div>
                    </div>
                </div>
                <div class="col-sm-6 col-lg-3">
                    <div class="card">
                        <div class="card-body">
                            <div class="subheader"><?php echo e(__('Most Active Day')); ?></div>
                            <div class="h1 mb-0" id="mostActiveDay">-</div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Events Table -->
            <div class="card mb-3" id="eventsTableCard">
                <div class="card-header">
                    <h3 class="card-title"><?php echo e(__('Harsh Driving Events')); ?></h3>
                </div>
                <div class="card-body p-0">
                    <div id="events-table"></div>
                </div>
            </div>

            <!-- Charts Section -->
            <div class="row row-cards mb-3 d-none" id="chartsSection">
                <!-- Event Trends Chart -->
                <div class="col-lg-6">
                    <div class="card">
                        <div class="card-header">
                            <h3 class="card-title"><?php echo e(__('Event Trends by Day')); ?></h3>
                        </div>
                        <div class="card-body">
                            <div id="eventTrendsChart" style="height: 300px;"></div>
                        </div>
                    </div>
                </div>
                
                <!-- Severity Distribution Chart -->
                <div class="col-lg-6">
                    <div class="card">
                        <div class="card-header">
                            <h3 class="card-title"><?php echo e(__('Severity Distribution')); ?></h3>
                        </div>
                        <div class="card-body">
                            <div id="severityChart" style="height: 300px;"></div>
                        </div>
                    </div>
                </div>
                
                <!-- Time of Day Analysis -->
                <div class="col-lg-6">
                    <div class="card">
                        <div class="card-header">
                            <h3 class="card-title"><?php echo e(__('Events by Hour of Day')); ?></h3>
                        </div>
                        <div class="card-body">
                            <div id="hourlyChart" style="height: 300px;"></div>
                        </div>
                    </div>
                </div>
                
                <!-- Vehicle Comparison -->
                <div class="col-lg-6">
                    <div class="card">
                        <div class="card-header">
                            <h3 class="card-title"><?php echo e(__('Events by Vehicle')); ?></h3>
                        </div>
                        <div class="card-body">
                            <div id="vehicleChart" style="height: 300px;"></div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Map Visualization -->
            <div class="card mb-3 d-none" id="mapCard">
                <div class="card-header">
                    <h3 class="card-title"><?php echo e(__('Event Locations Map')); ?></h3>
                </div>
                <div class="card-body p-0">
                    <div id="events-map" style="height: 500px;"></div>
                </div>
            </div>

            <!-- Loading State -->
            <div class="text-center py-4 d-none" id="loadingState">
                <div class="spinner-border text-primary" role="status">
                    <span class="visually-hidden"><?php echo e(__('Loading...')); ?></span>
                </div>
                <div class="mt-2"><?php echo e(__('Generating report...')); ?></div>
            </div>

            <!-- No Data State -->
            <div class="text-center py-4 d-none" id="noDataState">
                <div class="empty">
                    <div class="empty-img">
                        <i class="ti ti-arrow-autofit-content text-muted" style="font-size: 3rem;"></i>
                    </div>
                    <p class="empty-title"><?php echo e(__('No harsh driving events found')); ?></p>
                    <p class="empty-subtitle text-muted">
                        <?php echo e(__('No harsh driving events were detected for the selected period and limits.')); ?>

                    </p>
                </div>
            </div>

            <!-- No Vehicle Selected State -->
            <div class="text-center py-4 d-none" id="noVehicleState">
                <div class="empty">
                    <div class="empty-img">
                        <i class="ti ti-car text-muted" style="font-size: 3rem;"></i>
                    </div>
                    <p class="empty-title"><?php echo e(__('No Vehicle Selected')); ?></p>
                    <p class="empty-subtitle text-muted">
                        <?php echo e(__('Please select a vehicle from the dropdown above to generate the harsh driving analysis report.')); ?>

                    </p>
                </div>
            </div>
        </div>
    </div>
</div>
<?php $__env->stopSection(); ?>

<?php $__env->startSection('styles'); ?>
<style>
    #events-map .map-icon-label {
        padding: 0px !important;
        border-width: 0px !important;
        font-size: 15px !important;
    }
</style>
<?php $__env->stopSection(); ?>

<?php $__env->startSection('scripts'); ?>
<script>

    let reportData = [];
    let eventsTable;
    let vehicleList = [];
    let myMap;
    let event_groups = [];
    var speedUnit = '';
    var lengthUnit = '';
    var liquidUnit = '';
    var currency = '';

    $('#summaryCards').addClass('d-none');
    $('#eventsTableCard').addClass('d-none');
    $('#mapCard').addClass('d-none');
    $('#loadingState').removeClass('d-none');
    $('#noDataState').addClass('d-none');
    $('#noVehicleState').addClass('d-none');

    onAppReady(function() {        
        loadVehicles()
        .then(initializeEventsTable)
        .then(generateReport);
        
    });
    
    async function loadVehicles() {
        http.post({
            url: '/api/vehicles-list-with-alarm',
            data: { alarm_id: 305 }, // Acceleration/Deceleration alarm ID
            loading: false
        }).then(function(response) {
            if (response.success && response.data.vehicles) {
                const select = $('#vehicleSelect');
                select.empty();
                select.append('<option value=""><?php echo e(__('All Vehicles')); ?></option>');
                const vehiclesWithAlarm = [];
                const vehiclesWithoutAlarm = [];
                vehicleList = response.data.vehicles;
                response.data.vehicles.forEach(vehicle => {
                    if (vehicle.has_alarm) {
                        vehiclesWithAlarm.push(vehicle);
                    } else {
                        vehiclesWithoutAlarm.push(vehicle);
                    }
                });
                if (vehiclesWithAlarm.length > 0) {
                    select.append('<optgroup label="<?php echo e(__('Vehicles with Harsh Driving Alarm')); ?>">');
                    vehiclesWithAlarm.forEach(vehicle => {
                        var settings = JSON.parse(vehicle.alarm_settings);
                        var acc_limit = settings.p1 || 15;
                        var dec_limit = settings.p2 || 15;
                        select.append(`<option value="${vehicle.did}" data-acc-limit="${acc_limit}" data-dec-limit="${dec_limit}">${vehicle.name} (${acc_limit}/${dec_limit}) ✓</option>`);
                    });
                    select.append('</optgroup>');
                }
                if (vehiclesWithoutAlarm.length > 0) {
                    select.append('<optgroup label="<?php echo e(__('Vehicles without Harsh Driving Alarm')); ?>">');
                    vehiclesWithoutAlarm.forEach(vehicle => {
                        select.append(`<option value="${vehicle.did}" data-acc-limit="15" data-dec-limit="15">${vehicle.name} ✗</option>`);
                    });
                    select.append('</optgroup>');
                }
                if (vehiclesWithoutAlarm.length > 0) {
                    $('#vehicleSettingsLink').removeClass('d-none');
                } else {
                    $('#vehicleSettingsLink').addClass('d-none');
                }
                if (vehiclesWithAlarm.length === 0) {
                    showConfirmDialog({
                        title: '<?php echo e(__('Warning')); ?>',
                        message: '<?php echo e(__('No vehicles have harsh driving alarm enabled. Please enable harsh driving alarm for at least one vehicle to use this report.')); ?>',
                        confirmText: '<?php echo e(__('Go to Settings')); ?>',
                        cancelText: '<?php echo e(__('Cancel')); ?>',
                        type: 'warning',
                        onConfirm: function() {
                            window.location.href = '/vehicles-list';
                        },
                        onCancel: function() {}
                    });
                }
            }
        }).catch(function(error) {
            console.error('<?php echo e(__('Error loading vehicles')); ?>:', error);
        });
    }

    function goToVehicleSettings() {
        window.location.href = '/vehicles-list';
    }

    // Auto-populate acceleration limit when vehicle is selected
    $('#vehicleSelect').on('change', function() {
        const selectedOption = $(this).find('option:selected');
        const accLimit = selectedOption.data('acc-limit');
        
        if (accLimit) {
            $('#accelerationLimit').val(accLimit);
        }
    });

    async function generateReport() {
        const vehicleId = $('#vehicleSelect').val();
        const startDate = $('#startDate').val();
        const endDate = $('#endDate').val();
        const accelerationLimit = $('#accelerationLimit').val();

        if (!startDate || !endDate) {
            showConfirmDialog({
                title: '<?php echo e(__("Warning")); ?>',
                message: '<?php echo e(__("Please select start and end dates")); ?>',
                confirmText: '<?php echo e(__("OK")); ?>',
                showCancelButton: false,
                type: 'warning',
                onConfirm: function() {},
                onCancel: function() {}
            });
            return;
        }

        if (!startDate || !endDate) {
            showConfirmDialog({
                title: '<?php echo e(__("Warning")); ?>',
                message: '<?php echo e(__("Please select start and end dates")); ?>',
                confirmText: '<?php echo e(__("OK")); ?>',
                showCancelButton: false,
                type: 'warning',
                onConfirm: function() {},
                onCancel: function() {}
            });
            return;
        }

        // Show loading state
        showLoadingState();

        const requestData = {
            start_date: startDate,
            end_date: endDate
        };
        
        // Only add device_id if a specific vehicle is selected
        if (vehicleId) {
            requestData.device_id = vehicleId;
        }

        http.post({
            url: '/api/reports/harsh-driving',
            data: requestData,
            loading: false
        })
        .then(data => {
            hideLoadingState();
            
            if (data.success) {
                // Set units after data is loaded
                speedUnit = __(sessionSettings.speedUnit || 'km/h');
                lengthUnit = __(sessionSettings.lengthUnit || 'km');
                liquidUnit = __(sessionSettings.liquidUnit || 'lt');
                currency = __(sessionSettings.currency || 'TRY');

                reportData = data.data;

                var message = __('#name# used #value# records').replace('#name#', __('Harsh Driving')).replace('#value#', reportData.length);
                showReportDataProcessingInfo(message);

                processHarshDrivingData(data.data.filter(c => c.speed > 0));
            } else {
                showNoDataState();
                showConfirmDialog({
                    title: '<?php echo e(__("Error")); ?>',
                    message: data.message || '<?php echo e(__("Unknown error occurred")); ?>',
                    confirmText: '<?php echo e(__("OK")); ?>',
                    showCancelButton: false,
                    type: 'danger',
                    onConfirm: function() {},
                    onCancel: function() {}
                });
            }
        })
        .catch(err => {
            hideLoadingState();
            showNoDataState();
            var err = err;
            showConfirmDialog({
                title: '<?php echo e(__("Error")); ?>',
                message: __(err.message),
                confirmText: '<?php echo e(__("OK")); ?>',
                showCancelButton: false,
                type: 'danger',
                onConfirm: function() {},
                onCancel: function() {}
            });
        });
    }

    // Process harsh driving data
    function processHarshDrivingData(data) {
        if (!data || data.length === 0) {
            showNoDataState();
            return;
        }

        // Analyze acceleration/deceleration patterns
        const accelerationLimit = parseFloat($('#accelerationLimit').val()) || 15;
        const events = analyzeHarshDrivingEvents(data, accelerationLimit);

        if (events.length === 0) {
            showNoDataState();
            return;
        }

        // Calculate summary statistics
        const summary = calculateEventsSummary(events);
        
        // Display the processed data
        displayHarshDrivingReport({
            summary: summary,
            events: events,
            acceleration_limit: accelerationLimit
        });
    }

    // Analyze harsh driving events from GPS data
    function analyzeHarshDrivingEvents(data, accelerationLimit) {
        const events = [];
        
        for (let i = 1; i < data.length; i++) {
            const current = data[i];
            const previous = data[i - 1];
            
            // Calculate time difference in seconds
            const timeDiff = (new Date(current.timestamp) - new Date(previous.timestamp)) / 1000;
            
            if (timeDiff > 0) {
                // Calculate speed difference
                const speedDiff = parseFloat(current.speed) - parseFloat(previous.speed);
                
                // Calculate acceleration in km/s²
                const acceleration = speedDiff / timeDiff;
                
                // Check if this is a harsh driving event
                if (Math.abs(acceleration) > accelerationLimit) {
                    events.push({
                        id: current.id,
                        timestamp: current.timestamp,
                        latitude: current.latitude,
                        longitude: current.longitude,
                        speed: parseFloat(current.speed),
                        previous_speed: parseFloat(previous.speed),
                        acceleration: acceleration,
                        event_type: acceleration > 0 ? 'acceleration' : 'deceleration',
                        severity: Math.abs(acceleration) > accelerationLimit * 1.5 ? 'high' : 'medium',
                        time_diff: timeDiff,
                        device_id: current.device_id,
                        direction: current.direction
                    });
                }
            }
        }
        
        return events;
    }

    // Calculate summary statistics for harsh driving events
    function calculateEventsSummary(events) {
        const accelerationEvents = events.filter(e => e.event_type === 'acceleration');
        const decelerationEvents = events.filter(e => e.event_type === 'deceleration');
        
        // Group events by date
        const dateCounts = {};
        events.forEach(event => {
            const date = event.timestamp.split(' ')[0]; // Get date part
            dateCounts[date] = (dateCounts[date] || 0) + 1;
        });
        
        // Find most active date
        let mostActiveDate = null;
        if (Object.keys(dateCounts).length > 0) {
            const maxDate = Object.keys(dateCounts).reduce((a, b) => dateCounts[a] > dateCounts[b] ? a : b);
            mostActiveDate = {
                date: maxDate,
                events: dateCounts[maxDate]
            };
        }
        
        // Count unique vehicles if multiple vehicles are involved
        const uniqueVehicles = [...new Set(events.map(e => e.device_id))];
        
        return {
            total_events: events.length,
            acceleration_events: accelerationEvents.length,
            deceleration_events: decelerationEvents.length,
            most_active_date: mostActiveDate,
            unique_vehicles: uniqueVehicles.length
        };
    }

    // Display harsh driving report
    function displayHarshDrivingReport(data) {
        // Show all sections
        $('#summaryCards').removeClass('d-none');
        $('#eventsTableCard').removeClass('d-none');
        $('#chartsSection').removeClass('d-none');
        $('#mapCard').removeClass('d-none');
        $('#noDataState').addClass('d-none');
        $('#noVehicleState').addClass('d-none');

        // Update summary cards
        updateSummary(data.summary);

        // Update table
        updateTable(data.events);

        // Update charts
        updateCharts(data.events);

        // Update map
        updateMap(data.events);
    }

    function updateSummary(summary) {
        $('#totalEvents').text(summary.total_events);
        $('#accelerationEvents').text(summary.acceleration_events);
        $('#decelerationEvents').text(summary.deceleration_events);
        
        if (summary.most_active_date) {
            let mostActiveText = summary.most_active_date.events + ' <?php echo e(__("events")); ?>';
            if (summary.unique_vehicles > 1) {
                mostActiveText += ` (${summary.unique_vehicles} <?php echo e(__("vehicles")); ?>)`;
            }
            $('#mostActiveDay').text(mostActiveText);
        } else {
            $('#mostActiveDay').text('-');
        }
    }

    function updateTable(events) {
        // Table should already be initialized, but check just in case
        if (!eventsTable) {
            initializeEventsTable();
        }
        
        const tableData = events.map((event, index) => ({
            id: index + 1,
            date: event.timestamp.split(' ')[0],
            time: event.timestamp.split(' ')[1],
            vehicle: vehicleList.find(v => v.did === event.device_id)?.name || `Vehicle ${event.device_id}`,
            event_type: event.event_type,
            acceleration: Math.round(event.acceleration * 100) / 100,
            speed: Math.round(event.speed),
            previous_speed: Math.round(event.previous_speed),
            severity: event.severity,
            location: `${event.latitude}, ${event.longitude}`,
            direction: event.direction || '-',
            latitude: event.latitude,
            longitude: event.longitude
        }));

        setTimeout(() => {
            eventsTable.setData(tableData);
        }, 1000);
    }

    function updateMap(events) {
        if (!myMap) {
            initializeMap();
        }
        
        // Clear existing markers
        myMap.removeMarkers();
        myMap.clearAllPolygons();
        myMap.removeCircles();
        myMap.removePolyline();
        
        if (!events || events.length === 0) {
            return;
        }
        
        // Add regions and places to the map
        addRegionsAndPlacesToMap();
        
        // Create markers for each event
        const markerModels = [];
        
        events.forEach((event, index) => {
            const lat = parseFloat(event.latitude);
            const lng = parseFloat(event.longitude);
            
            if (!isNaN(lat) && !isNaN(lng)) {
                let markerIcon = "ti ti-arrow-up";
                let markerColor = "#28a745"; // Green for acceleration
                let markerLabel = "";
                
                if (event.event_type === 'deceleration') {
                    markerIcon = "ti ti-arrow-down";
                    markerColor = "#dc3545"; // Red for deceleration
                }
                
                if (event.severity === 'high') {
                    markerLabel = `<div class="badge" style="background-color: ${markerColor}; color: white;"><?php echo e(__("HIGH")); ?></div>`;
                } else {
                    markerLabel = `<div class="badge" style="background-color: ${markerColor}; color: white;">${Math.round(Math.abs(event.acceleration))}</div>`;
                }
                
                // Create marker
                markerModels.push({
                    id: `event-${event.id}-${index}`,
                    markerClassName: "harsh-driving-marker",
                    label: markerLabel,
                    acceleration: Math.round(event.acceleration * 100) / 100,
                    device_id: event.device_id,
                    popup: `
                        <strong><?php echo e(__("Harsh Driving Event")); ?> #${index + 1}</strong><br>
                        <?php echo e(__("Date")); ?>: ${dateTimeFormatter(event.timestamp.split(' ')[0])}<br>
                        <?php echo e(__("Time")); ?>: ${event.timestamp.split(' ')[1]}<br>
                        <?php echo e(__("Type")); ?>: ${event.event_type === 'acceleration' ? '<?php echo e(__("Acceleration")); ?>' : '<?php echo e(__("Deceleration")); ?>'}<br>
                        <?php echo e(__("Acceleration")); ?>: ${Math.round(event.acceleration * 100) / 100} km/s²<br>
                        <?php echo e(__("Speed")); ?>: ${Math.round(event.speed)} km/h<br>
                        <?php echo e(__("Previous Speed")); ?>: ${Math.round(event.previous_speed)} km/h<br>
                        <?php echo e(__("Severity")); ?>: ${event.severity === 'high' ? '<?php echo e(__("High")); ?>' : '<?php echo e(__("Medium")); ?>'}<br>
                        <?php echo e(__("Direction")); ?>: ${event.direction || '-'}°
                    `,
                    coord: [lat, lng],
                    icon: {
                        name: markerIcon,
                        color: markerColor,
                        direction: 0,
                        className: "harsh-driving-marker",
                    }
                });
            }
        });
        
        // Add markers to map
        if (markerModels.length > 0) {
            myMap.pinMarkers(markerModels);

            // Fit map to show all markers
            setTimeout(() => { myMap.fitBounds(); } , 1000);
        }
    }

    // Initialize map
    function initializeMap() {
        myMap = new LeafletMap();
        myMap.initMap("events-map", {
            // Default zoom level
        });
    }

    // Initialize table
    async function initializeEventsTable() {
        eventsTable = createTabulator("#events-table", {
            initialSort: [
                {column: "date", dir: "desc"},
                {column: "time", dir: "desc"}
            ],
            columns: [
                {title: "#", field: "id", width: 50, hozAlign: "center"},
                {
                    title: "<?php echo e(__('Date')); ?>", 
                    field: "date",
                    formatter: function(cell) {
                        return dateTimeFormatter(cell.getValue());
                    }
                },
                {
                    title: "<?php echo e(__('Time')); ?>", 
                    field: "time",
                    formatter: function(cell) {
                        return formatTime(cell.getValue());
                    }
                },
                {
                    title: "<?php echo e(__('Vehicle')); ?>", 
                    field: "vehicle"
                },
                {
                    title: "<?php echo e(__('Type')); ?>", 
                    field: "event_type", 
                    hozAlign: "center",
                    formatter: function(cell) {
                        const type = cell.getValue();
                        if (type === 'acceleration') {
                            return `<span class="badge text-white bg-success"><?php echo e(__('Acceleration')); ?></span>`;
                        } else {
                            return `<span class="badge text-white bg-danger"><?php echo e(__('Deceleration')); ?></span>`;
                        }
                    }
                },
                {
                    title: "<?php echo e(__('Acceleration')); ?>", 
                    field: "acceleration", 
                    hozAlign: "center",
                    formatter: function(cell) {
                        const acc = cell.getValue();
                        const severity = cell.getData().severity;
                        
                        if (severity === 'high') {
                            return `<span class="badge text-white bg-danger">${acc} km/s²</span>`;
                        } else {
                            return `<span class="badge text-white bg-warning">${acc} km/s²</span>`;
                        }
                    }
                },
                {
                    title: "<?php echo e(__('Speed')); ?>", 
                    field: "speed", 
                    hozAlign: "center",
                    formatter: function(cell) {
                        return `${cell.getValue()} ${speedUnit}`;
                    }
                },
                {
                    title: "<?php echo e(__('Previous Speed')); ?>", 
                    field: "previous_speed", 
                    hozAlign: "center",
                    formatter: function(cell) {
                        return `${cell.getValue()} ${speedUnit}`;
                    }
                },
                {
                    title: "<?php echo e(__('Severity')); ?>", 
                    field: "severity", 
                    hozAlign: "center",
                    formatter: function(cell) {
                        const severity = cell.getValue();
                        if (severity === 'high') {
                            return `<span class="badge text-white bg-danger"><?php echo e(__('High')); ?></span>`;
                        } else {
                            return `<span class="badge text-white bg-warning"><?php echo e(__('Medium')); ?></span>`;
                        }
                    }
                },
                {
                    title: "<?php echo e(__('Direction')); ?>", 
                    field: "direction", 
                    hozAlign: "center",
                    formatter: function(cell) {
                        const direction = cell.getValue();
                        if (direction && direction !== '-') {
                            return `${direction}°`;
                        }
                        return '-';
                    }
                },
                {
                    title: "<?php echo e(__('Location')); ?>", 
                    field: "location", 
                    formatter: function(cell) {
                        const data = cell.getData();
                        if (data.latitude && data.longitude) {
                            return `<a href="#" class="btn btn-sm btn-outline-primary map-link" onclick="showLocationOnMap(${data.latitude}, ${data.longitude}); return false;"><?php echo e(__('See on map')); ?></a>`;
                        }
                        return '-';
                    }
                }
            ]
        });
    }

    // Add regions and places to the map
    function addRegionsAndPlacesToMap() {
        // Load regions data
        http.post({
            url: '/regions-list',
            loading: false
        }).then(function(regionsResponse) {
            if (regionsResponse.success && regionsResponse.data) {
                const regions = regionsResponse.data;
                
                // Add regions to map
                regions.forEach(function(region) {
                    if (region.polygon_json) {
                        myMap.addPolygon({
                            id: region.id,
                            label: region.name,
                            color: "gray",
                            fillColor: "gray",
                            coords: region.polygon_json,
                        });
                    }
                });
            }
        }).catch(function(error) {
            console.error('Error loading regions:', error);
        });

        // Load places data
        http.post({
            url: '/places-list',
            loading: false
        }).then(function(placesResponse) {
            if (placesResponse.success && placesResponse.data) {
                const places = placesResponse.data;
                
                // Add places to map
                var placeMarkerColor = "gray";
                places.forEach(function(place) {
                    if (place.latitude && place.longitude) {
                        var marker = [
                            {
                                id: place.id,
                                markerClassName: "place-marker",
                                device_id: 0,
                                label: place.name,
                                status: place.status_id,
                                popup: place.name,
                                coord: [
                                    parseFloat(place.latitude),
                                    parseFloat(place.longitude),
                                ],
                                icon: {
                                    name: "fa-solid fa-location-dot",
                                    color: placeMarkerColor,
                                    direction: 0,
                                    className: "",
                                },
                            },
                        ];
                        myMap.pinMarkers(marker);
                        
                        // Add circle if radius is defined
                        if (place.radius && place.radius > 0) {
                            var circle = L.circle([place.latitude, place.longitude], {
                                radius: place.radius,
                                color: placeMarkerColor,
                                fillColor: placeMarkerColor,
                                fillOpacity: 0.2,
                            }).addTo(myMap.mapLet);
                        }
                    }
                });
            }
        }).catch(function(error) {
            console.error('Error loading places:', error);
        });
    }

    function showLocationOnMap(lat, lng) {
        if (!lat || !lng || isNaN(lat) || isNaN(lng)) {
            showConfirmDialog({
                title: '<?php echo e(__("Error")); ?>',
                message: '<?php echo e(__("Invalid coordinates")); ?>',
                confirmText: '<?php echo e(__("OK")); ?>',
                showCancelButton: false,
                type: 'danger',
                onConfirm: function() {},
                onCancel: function() {}
            });
            return;
        }
        
        // Prevent any default scrolling behavior
        event.preventDefault();
        
        // Center map on the location with smooth animation
        myMap.mapLet.setView([parseFloat(lat), parseFloat(lng)], 16, {
            animate: true,
            duration: 0.5
        });
        
        // Add a temporary marker to highlight the location
        const tempMarker = L.marker([lat, lng], {
            icon: L.divIcon({
                className: 'temp-location-marker',
                html: '<i class="fa-solid fa-crosshairs" style="color: red; font-size: 20px;"></i>',
                iconSize: [20, 20],
                iconAnchor: [10, 10]
            })
        }).addTo(myMap.mapLet);
        
        // Remove the temporary marker after 3 seconds
        setTimeout(() => {
            myMap.mapLet.removeLayer(tempMarker);
        }, 3000);
        
        // Ensure map container is visible and focused
        $('#mapCard').removeClass('d-none');
        $('#events-map').focus();
    }

    // Show loading state
    function showLoadingState() {
        $('#loadingState').removeClass('d-none');
        $('#summaryCards').addClass('d-none');
        $('#eventsTableCard').addClass('d-none');
        $('#chartsSection').addClass('d-none');
        $('#mapCard').addClass('d-none');
        $('#noDataState').addClass('d-none');
        $('#noVehicleState').addClass('d-none');
    }

    // Hide loading state
    function hideLoadingState() {
        $('#loadingState').addClass('d-none');
    }

    // Show no data state
    function showNoDataState() {
        $('#noDataState').removeClass('d-none');
        $('#summaryCards').addClass('d-none');
        $('#eventsTableCard').addClass('d-none');
        $('#chartsSection').addClass('d-none');
        $('#mapCard').addClass('d-none');
        $('#noVehicleState').addClass('d-none');
    }

    // Update charts with event data
    function updateCharts(events) {
        if (!events || events.length === 0) return;
        
        // Create event trends chart
        createEventTrendsChart(events);
        
        // Create severity distribution chart
        createSeverityChart(events);
        
        // Create hourly analysis chart
        createHourlyChart(events);
        
        // Create vehicle comparison chart
        createVehicleChart(events);
    }

    // Create event trends chart (line chart)
    function createEventTrendsChart(events) {
        // Check if ApexCharts is available
        if (typeof ApexCharts === 'undefined') {
            console.warn('ApexCharts is not loaded');
            return;
        }

        // Group events by date
        const dateGroups = {};
        events.forEach(event => {
            const date = event.timestamp.split(' ')[0];
            if (!dateGroups[date]) {
                dateGroups[date] = {
                    acceleration: 0,
                    deceleration: 0
                };
            }
            if (event.event_type === 'acceleration') {
                dateGroups[date].acceleration++;
            } else {
                dateGroups[date].deceleration++;
            }
        });

        const dates = Object.keys(dateGroups).sort();
        const accelerationData = dates.map(date => dateGroups[date].acceleration);
        const decelerationData = dates.map(date => dateGroups[date].deceleration);

        const options = {
            series: [
                {
                    name: '<?php echo e(__("Acceleration")); ?>',
                    data: accelerationData
                },
                {
                    name: '<?php echo e(__("Deceleration")); ?>',
                    data: decelerationData
                }
            ],
            chart: {
                type: 'line',
                height: 300,
                toolbar: {
                    show: false
                }
            },
            colors: ['#28a745', '#dc3545'],
            dataLabels: {
                enabled: false
            },
            stroke: {
                curve: 'smooth',
                width: 3
            },
            xaxis: {
                categories: dates.map(date => dateTimeFormatter(date)),
                title: {
                    text: '<?php echo e(__("Date")); ?>'
                }
            },
            yaxis: {
                title: {
                    text: '<?php echo e(__("Number of Events")); ?>'
                }
            },
            legend: {
                position: 'top'
            },
            tooltip: {
                y: {
                    formatter: function (val) {
                        return val + ' <?php echo e(__("events")); ?>';
                    }
                }
            }
        };

        // Safely destroy existing chart
        if (window.eventTrendsChart && typeof window.eventTrendsChart.destroy === 'function') {
            window.eventTrendsChart.destroy();
        }
        
        const chartElement = document.querySelector("#eventTrendsChart");
        if (chartElement) {
            window.eventTrendsChart = new ApexCharts(chartElement, options);
            window.eventTrendsChart.render();
        }
    }

    // Create severity distribution chart (pie chart)
    function createSeverityChart(events) {
        // Check if ApexCharts is available
        if (typeof ApexCharts === 'undefined') {
            console.warn('ApexCharts is not loaded');
            return;
        }

        const severityCounts = {
            high: 0,
            medium: 0
        };

        events.forEach(event => {
            severityCounts[event.severity]++;
        });

        const options = {
            series: [severityCounts.high, severityCounts.medium],
            chart: {
                type: 'pie',
                height: 300
            },
            labels: ['<?php echo e(__("High")); ?>', '<?php echo e(__("Medium")); ?>'],
            colors: ['#dc3545', '#ffc107'],
            legend: {
                position: 'bottom'
            },
            tooltip: {
                y: {
                    formatter: function (val) {
                        return val + ' <?php echo e(__("events")); ?>';
                    }
                }
            },
            plotOptions: {
                pie: {
                    donut: {
                        size: '60%'
                    }
                }
            }
        };

        // Safely destroy existing chart
        if (window.severityChart && typeof window.severityChart.destroy === 'function') {
            window.severityChart.destroy();
        }
        
        const chartElement = document.querySelector("#severityChart");
        if (chartElement) {
            window.severityChart = new ApexCharts(chartElement, options);
            window.severityChart.render();
        }
    }

    // Create hourly analysis chart (bar chart)
    function createHourlyChart(events) {
        // Check if ApexCharts is available
        if (typeof ApexCharts === 'undefined') {
            console.warn('ApexCharts is not loaded');
            return;
        }

        const hourlyCounts = {};
        
        // Initialize all hours
        for (let i = 0; i < 24; i++) {
            hourlyCounts[i] = 0;
        }

        events.forEach(event => {
            const hour = parseInt(event.timestamp.split(' ')[1].split(':')[0]);
            hourlyCounts[hour]++;
        });

        const hours = Object.keys(hourlyCounts).map(h => h + ':00');
        const counts = Object.values(hourlyCounts);

        const options = {
            series: [{
                name: '<?php echo e(__("Events")); ?>',
                data: counts
            }],
            chart: {
                type: 'bar',
                height: 300,
                toolbar: {
                    show: false
                }
            },
            colors: ['#007bff'],
            plotOptions: {
                bar: {
                    horizontal: false,
                    columnWidth: '70%',
                    endingShape: 'rounded'
                }
            },
            dataLabels: {
                enabled: false
            },
            xaxis: {
                categories: hours,
                title: {
                    text: '<?php echo e(__("Hour of Day")); ?>'
                }
            },
            yaxis: {
                title: {
                    text: '<?php echo e(__("Number of Events")); ?>'
                }
            },
            tooltip: {
                y: {
                    formatter: function (val) {
                        return val + ' <?php echo e(__("events")); ?>';
                    }
                }
            }
        };

        // Safely destroy existing chart
        if (window.hourlyChart && typeof window.hourlyChart.destroy === 'function') {
            window.hourlyChart.destroy();
        }
        
        const chartElement = document.querySelector("#hourlyChart");
        if (chartElement) {
            window.hourlyChart = new ApexCharts(chartElement, options);
            window.hourlyChart.render();
        }
    }

    // Create vehicle comparison chart (horizontal bar chart)
    function createVehicleChart(events) {
        // Check if ApexCharts is available
        if (typeof ApexCharts === 'undefined') {
            console.warn('ApexCharts is not loaded');
            return;
        }

        const vehicleCounts = {};
        
        events.forEach(event => {
            const vehicleName = vehicleList.find(v => v.did === event.device_id)?.name || `Vehicle ${event.device_id}`;
            if (!vehicleCounts[vehicleName]) {
                vehicleCounts[vehicleName] = {
                    acceleration: 0,
                    deceleration: 0
                };
            }
            if (event.event_type === 'acceleration') {
                vehicleCounts[vehicleName].acceleration++;
            } else {
                vehicleCounts[vehicleName].deceleration++;
            }
        });

        const vehicles = Object.keys(vehicleCounts);
        const accelerationData = vehicles.map(v => vehicleCounts[v].acceleration);
        const decelerationData = vehicles.map(v => vehicleCounts[v].deceleration);

        const options = {
            series: [
                {
                    name: '<?php echo e(__("Acceleration")); ?>',
                    data: accelerationData
                },
                {
                    name: '<?php echo e(__("Deceleration")); ?>',
                    data: decelerationData
                }
            ],
            chart: {
                type: 'bar',
                height: 300,
                stacked: true,
                toolbar: {
                    show: false
                }
            },
            colors: ['#28a745', '#dc3545'],
            plotOptions: {
                bar: {
                    horizontal: true,
                    dataLabels: {
                        total: {
                            enabled: true,
                            offsetX: 0,
                            style: {
                                fontSize: '13px',
                                fontWeight: 900
                            }
                        }
                    }
                }
            },
            stroke: {
                width: 1,
                colors: ['#fff']
            },
            xaxis: {
                categories: vehicles,
                title: {
                    text: '<?php echo e(__("Number of Events")); ?>'
                }
            },
            yaxis: {
                title: {
                    text: '<?php echo e(__("Vehicle")); ?>'
                }
            },
            tooltip: {
                shared: false,
                y: {
                    formatter: function (val) {
                        return val + ' <?php echo e(__("events")); ?>';
                    }
                }
            },
            legend: {
                position: 'top'
            }
        };

        // Safely destroy existing chart
        if (window.vehicleChart && typeof window.vehicleChart.destroy === 'function') {
            window.vehicleChart.destroy();
        }
        
        const chartElement = document.querySelector("#vehicleChart");
        if (chartElement) {
            window.vehicleChart = new ApexCharts(chartElement, options);
            window.vehicleChart.render();
        }
    }

    // Utility functions
    function formatDate(dateString) {
        return new Date(dateString).toLocaleDateString();
    }

    function formatTime(timeString) {
        return timeString;
    }
</script>
<?php $__env->stopSection(); ?>
<?php echo $__env->make('layouts.app', \Illuminate\Support\Arr::except(get_defined_vars(), ['__data', '__path']))->render(); ?><?php /**PATH C:\xampp\htdocs\aractakipv1\resources\views\reports\harsh-driving.blade.php ENDPATH**/ ?>