@extends('layouts.app')

@section('title', __('Inspection Schedule Report'))

@section('content')
<div class="page-wrapper" id="inspection-schedule-report">
    <div class="page-header d-print-none">
        <div class="container-xl">
            <div class="row g-2 align-items-center">
                <div class="col">
                    <div class="page-pretitle">
                        <a href="/reports" class="text-muted">{{ __('Reports') }}</a> / {{ __('Maintenance Reports') }}
                    </div>
                    <h2 class="page-title">
                        <i class="ti ti-calendar-exclamation me-2"></i>
                        {{ __('Inspection Schedule Report') }}
                    </h2>
                    <div class="text-muted mt-1">
                        {{ __('Monitor vehicle inspection due dates and compliance status') }}
                    </div>
                </div>
                <div class="col-auto ms-auto d-print-none">
                    <div class="btn-list">

                        <button type="button" class="btn btn-secondary" onclick="printReport()">
                            <i class="ti ti-printer me-1"></i>
                            {{ __('Print') }}
                        </button>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <div class="page-body">
        <div class="container-xl">
            <!-- Filters -->
            <div class="card mb-3">
                <div class="card-header">
                    <h3 class="card-title">{{ __('Filters') }}</h3>
                </div>
                <div class="card-body">
                    <div class="row g-3">
                        <div class="col-md-3">
                            <label class="form-label">{{ __('Status') }}</label>
                            <select class="form-select" id="statusFilter">
                                <option value="">{{ __('All Status') }}</option>
                                <option value="overdue">{{ __('Overdue') }}</option>
                                <option value="due_soon">{{ __('Due Soon (30 days)') }}</option>
                                <option value="upcoming">{{ __('Upcoming') }}</option>
                                <option value="compliant">{{ __('Compliant') }}</option>
                            </select>
                        </div>

                        <div class="col-md-3">
                            <label class="form-label">&nbsp;</label>
                            <div>
                                <button type="button" class="btn btn-primary w-100" onclick="generateReport()">
                                    <i class="icon ti ti-search me-1"></i>
                                    {{ __('Generate Report') }}
                                </button>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Summary Cards -->
            <div class="row row-cards mb-3" id="summaryCards">
                <div class="col-sm-6 col-lg-3">
                    <div class="card">
                        <div class="card-body">
                            <div class="d-flex align-items-center">
                                <div class="subheader">{{ __('Total Vehicles') }}</div>
                            </div>
                            <div class="h1 mb-0" id="totalVehicles">-</div>
                            <div class="text-muted" id="totalVehiclesTrend">
                                {{ __('All vehicles in system') }}
                            </div>
                        </div>
                    </div>
                </div>
                <div class="col-sm-6 col-lg-3">
                    <div class="card">
                        <div class="card-body">
                            <div class="d-flex align-items-center">
                                <div class="subheader">{{ __('With Inspection Date') }}</div>
                            </div>
                            <div class="h1 mb-0" id="withInspectionDate">-</div>
                            <div class="text-muted" id="withInspectionDateTrend">
                                {{ __('Vehicles with inspection monitoring') }}
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            <div class="row row-cards mb-3">
                <div class="col-sm-6 col-lg-3">
                    <div class="card">
                        <div class="card-body">
                            <div class="d-flex align-items-center">
                                <div class="subheader">{{ __('Overdue') }}</div>
                            </div>
                            <div class="h1 mb-0 text-danger" id="overdueCount">-</div>
                            <div class="text-muted" id="overdueTrend">
                                {{ __('Vehicles past due date') }}
                            </div>
                        </div>
                    </div>
                </div>
                <div class="col-sm-6 col-lg-3">
                    <div class="card">
                        <div class="card-body">
                            <div class="d-flex align-items-center">
                                <div class="subheader">{{ __('Due Soon') }}</div>
                            </div>
                            <div class="h1 mb-0 text-warning" id="dueSoonCount">-</div>
                            <div class="text-muted" id="dueSoonTrend">
                                {{ __('Due within 30 days') }}
                            </div>
                        </div>
                    </div>
                </div>
                <div class="col-sm-6 col-lg-3">
                    <div class="card">
                        <div class="card-body">
                            <div class="d-flex align-items-center">
                                <div class="subheader">{{ __('Compliant') }}</div>
                            </div>
                            <div class="h1 mb-0 text-success" id="compliantCount">-</div>
                            <div class="text-muted" id="compliantTrend">
                                {{ __('Up to date inspections') }}
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Charts Row -->
            <div class="row mb-3" id="chartsRow">
                <div class="col-lg-6">
                    <div class="card">
                        <div class="card-header">
                            <h3 class="card-title">{{ __('Inspection Status Distribution') }}</h3>
                        </div>
                        <div class="card-body">
                            <div id="statusDistributionChart" style="height: 300px;"></div>
                        </div>
                    </div>
                </div>
                <div class="col-lg-6">
                    <div class="card">
                        <div class="card-header">
                            <h3 class="card-title">{{ __('Monthly Due Dates') }}</h3>
                        </div>
                        <div class="card-body">
                            <div id="monthlyDueChart" style="height: 300px;"></div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Inspection Schedule Table -->
            <div class="card d-none" id="inspectionTableCard">
                <div class="card-header">
                    <h3 class="card-title">{{ __('Inspection Schedule Details') }}</h3>
                    <div class="card-actions">
                        <div class="btn-list">
                            <button class="btn btn-sm" onclick="tableInspection.download('csv', 'inspection-schedule.csv')">
                                <i class="ti ti-download"></i> {{ __('CSV') }}
                            </button>
                            <button class="btn btn-sm" onclick="toggleColumns()">
                                <i class="ti ti-columns"></i> {{ __('Columns') }}
                            </button>
                        </div>
                    </div>
                </div>
                <div class="card-body p-0">
                    <div id="inspection-table"></div>
                </div>
            </div>

            <!-- Vehicles Without Inspection Date Table -->
            <div class="card d-none" id="noInspectionTableCard">
                <div class="card-header">
                    <h3 class="card-title">{{ __('Vehicles Without Inspection Date') }}</h3>
                    <div class="card-subtitle text-muted">{{ __('Vehicles that need inspection date configuration') }}</div>
                    <div class="card-actions">
                        <div class="btn-list">
                            <button class="btn btn-sm" onclick="tableNoInspection.download('csv', 'vehicles-without-inspection.csv')">
                                <i class="ti ti-download"></i> {{ __('CSV') }}
                            </button>
                        </div>
                    </div>
                </div>
                <div class="card-body p-0">
                    <div id="no-inspection-table"></div>
                </div>
            </div>

            <!-- Loading State -->
            <div class="text-center py-4 d-none" id="loadingState">
                <div class="spinner-border text-primary" role="status">
                    <span class="visually-hidden">{{ __('Loading...') }}</span>
                </div>
                <div class="mt-2">{{ __('Generating inspection schedule report...') }}</div>
            </div>

            <!-- No Data State -->
            <div class="text-center py-4 d-none" id="noDataState">
                <div class="empty">
                    <div class="empty-img">
                        <i class="ti ti-calendar-exclamation text-muted" style="font-size: 3rem;"></i>
                    </div>
                    <p class="empty-title">{{ __('No inspection data found') }}</p>
                    <p class="empty-subtitle text-muted">
                        {{ __('No vehicles with inspection monitoring enabled for the selected period.') }}
                    </p>
                </div>
            </div>

            <!-- No Monitoring State -->
            <div class="text-center py-4 d-none" id="noMonitoringState">
                <div class="empty">
                    <div class="empty-img">
                        <i class="ti ti-settings text-muted" style="font-size: 3rem;"></i>
                    </div>
                    <p class="empty-title">{{ __('No Inspection Monitoring') }}</p>
                    <p class="empty-subtitle text-muted">
                        {{ __('Please enable inspection monitoring for vehicles in vehicle settings to use this report.') }}
                    </p>
                    <div class="empty-action">
                        <a href="/vehicles-list" class="btn btn-primary">
                            <i class="ti ti-settings me-1"></i>
                            {{ __('Go to Vehicle Settings') }}
                        </a>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>
@endsection

@section('styles')
<style>
    .status-badge {
        padding: 2px 8px;
        border-radius: 12px;
        font-size: 11px;
        font-weight: 600;
    }

    .status-overdue {
        background-color: #ef4444;
        color: white;
    }

    .status-due-soon {
        background-color: #f59e0b;
        color: white;
    }

    .status-upcoming {
        background-color: #3b82f6;
        color: white;
    }

    .status-compliant {
        background-color: #10b981;
        color: white;
    }

    .days-overdue {
        color: #ef4444;
        font-weight: 600;
    }

    .days-remaining {
        color: #f59e0b;
        font-weight: 600;
    }

    .days-compliant {
        color: #10b981;
        font-weight: 600;
    }
</style>
@endsection

@section('scripts')
<script>
    let reportData = [];
    let tableInspection;
    let tableNoInspection;
    let vehicles = [];
    let categories = [];
    let allVehicles = [];
    let vehiclesWithInspection = [];
    let vehiclesWithoutInspection = [];
    
    // Chart variables
    let statusDistributionChart;
    let monthlyDueChart;
    
    var speedUnit = '';
    var lengthUnit = '';
    var currency = '';

    $('#summaryCards').addClass('d-none');
    $('#chartsRow').addClass('d-none');
    $('#inspectionTableCard').addClass('d-none');
    $('#loadingState').addClass('d-none');
    $('#noDataState').addClass('d-none');
    $('#noMonitoringState').removeClass('d-none');
    
    onAppReady(function() {
        // Initialize the page after app is ready
        initializePage();
    });

    function initializePage() {
        initializeTable();
        initializeNoInspectionTable();
        loadInspectionData();
    }

    // Initialize table
    function initializeTable() {
        tableInspection = createTabulator("#inspection-table", {
            pagination: "local",
            paginationSize: 20,
            initialSort: [
                {column: "days_remaining", dir: "asc"}
            ],
            columns: [
                {title: "#", field: "id", width: 50, hozAlign: "center"},
                {
                    title: "{{ __('Vehicle') }}", 
                    field: "vehicle_name",
                    formatter: function(cell) {
                        const data = cell.getData();
                        return `<strong>${data.vehicle_name}</strong><br><small class="text-muted">${data.plate || ''}</small>`;
                    }
                },
                {
                    title: "{{ __('Inspection Date') }}", 
                    field: "inspection_date",
                    formatter: function(cell) {
                        const date = cell.getValue();
                        if (!date) return '<span class="text-muted">-</span>';
                        return dateFormatter(date);
                    }
                },
                {
                    title: "{{ __('Status') }}", 
                    field: "status", 
                    hozAlign: "center",
                    formatter: function(cell) {
                        const status = cell.getValue();
                        const daysRemaining = cell.getData().days_remaining;
                        
                        let badgeClass = '';
                        let statusText = '';
                        
                        if (status === 'overdue') {
                            badgeClass = 'status-overdue';
                            statusText = '{{ __("Overdue") }}';
                        } else if (status === 'due_soon') {
                            badgeClass = 'status-due-soon';
                            statusText = '{{ __("Due Soon") }}';
                        } else if (status === 'upcoming') {
                            badgeClass = 'status-upcoming';
                            statusText = '{{ __("Upcoming") }}';
                        } else {
                            badgeClass = 'status-compliant';
                            statusText = '{{ __("Compliant") }}';
                        }
                        
                        return `<span class="status-badge ${badgeClass}">${statusText}</span>`;
                    }
                },
                {
                    title: "{{ __('Days') }}", 
                    field: "days_remaining", 
                    hozAlign: "center",
                    formatter: function(cell) {
                        const days = cell.getValue();
                        const status = cell.getData().status;
                        
                        let textClass = '';
                        let prefix = '';
                        
                        if (status === 'overdue') {
                            textClass = 'days-overdue';
                            prefix = '{{ __("Overdue by") }} ';
                        } else if (status === 'due_soon') {
                            textClass = 'days-remaining';
                            prefix = '{{ __("Due in") }} ';
                        } else if (status === 'upcoming') {
                            textClass = 'days-remaining';
                            prefix = '{{ __("Due in") }} ';
                        } else {
                            textClass = 'days-compliant';
                            prefix = '{{ __("Valid for") }} ';
                        }
                        
                        return `<span class="${textClass}">${prefix}${Math.abs(days)} {{ __("days") }}</span>`;
                    }
                },
                {
                    title: "{{ __('Alarm Status') }}", 
                    field: "alarm_status", 
                    hozAlign: "center",
                    formatter: function(cell) {
                        const status = cell.getValue();
                        if (status) {
                            return '<span class="badge text-white bg-success">{{ __("Active") }}</span>';
                        } else {
                            return '<span class="badge text-white bg-danger">{{ __("Inactive") }}</span>';
                        }
                    }
                },
                {
                    title: "{{ __('Actions') }}", 
                    field: "actions", 
                    hozAlign: "center",
                    formatter: function(cell) {
                        const data = cell.getData();
                        let buttons = '';
                        var urlModel = {
                            'did': data.vehicle_id
                        };
                        var urlData = Base64Helper.encode(urlModel);
                        // Go to vehicle settings button
                        if (data.vehicle_id) {
                            buttons += `<a href="/vehicle-settings/${urlData}" class="btn btn-sm btn-primary me-1" title="{{ __('Vehicle Settings') }}">
                                <i class="ti ti-settings"></i>
                                <span class="d-none d-md-inline">{{ __("Settings") }}</span>
                            </a>`;
                        }
                        
                        // Vehicle history button
                        if (data.vehicle_id) {
                            const today = new Date().toISOString().split('T')[0];
                            const urlEncoded = btoa(JSON.stringify({
                                did: data.vehicle_id,
                                date: today,
                                start_time: "00:01",
                                end_time: "23:59"
                            }));
                            buttons += `<a href="/vehicle-map/${urlEncoded}" class="btn btn-sm btn-info me-1" title="{{ __('Vehicle History') }}">
                                <i class="ti ti-history"></i>
                                <span class="d-none d-md-inline">{{ __("History") }}</span>
                            </a>`;
                        }
                        
                        return buttons;
                    }
                }
            ]
        });
    }

    // Initialize table for vehicles without inspection date
    function initializeNoInspectionTable() {
        tableNoInspection = createTabulator("#no-inspection-table", {
            initialSort: [
                {column: "vehicle_name", dir: "asc"}
            ],
            columns: [
                {title: "#", field: "id", width: 50, hozAlign: "center"},
                {
                    title: "{{ __('Vehicle') }}", 
                    field: "vehicle_name",
                    formatter: function(cell) {
                        const data = cell.getData();
                        return `<strong>${data.vehicle_name}</strong><br><small class="text-muted">${data.plate || ''}</small>`;
                    }
                },
                {
                    title: "{{ __('Inspection Monitoring') }}", 
                    field: "inspection_monitoring", 
                    hozAlign: "center",
                    formatter: function(cell) {
                        const status = cell.getValue();
                        if (status) {
                            return '<span class="badge text-white bg-warning">{{ __("Enabled (No Date)") }}</span>';
                        } else {
                            return '<span class="badge text-white bg-danger">{{ __("Disabled") }}</span>';
                        }
                    }
                },
                {
                    title: "{{ __('Actions') }}", 
                    field: "actions", 
                    hozAlign: "center",
                    formatter: function(cell) {
                        const data = cell.getData();
                        let buttons = '';

                        var urlModel = {
                            'did': data.vehicle_id
                        };
                        var urlData = Base64Helper.encode(urlModel);
                        
                        // Go to vehicle settings button
                        if (data.vehicle_id) {
                            buttons += `<a href="/vehicle-settings/${urlData}" class="btn btn-sm btn-primary me-1" title="{{ __('Vehicle Settings') }}">
                                <i class="ti ti-settings"></i>
                                <span class="d-none d-md-inline">{{ __("Settings") }}</span>
                            </a>`;
                        }
                        
                        return buttons;
                    }
                }
            ]
        });
    }

    // Load inspection data
    function loadInspectionData() {
        showLoadingState();


        http.post({
            url: '/api/reports/inspection-schedule',
            data: {},
            loading: false
        })
        .then(function(response) {
            hideLoadingState();
            
            if (response.success) {

        
                // Set units after data is loaded
                speedUnit = __(window.sessionSettings.speedUnit || 'km/h');
                lengthUnit = __(window.sessionSettings.lengthUnit || 'km');
                currency = __(window.sessionSettings.currency || 'TRY');
                
                reportData = response.data;
                allVehicles = response.data.vehicles || [];
                vehicles = response.data.vehicles || [];

                var message = __('#name# used #value# records').replace('#name#', __('Inspection Schedule')).replace('#value#', allVehicles.length);
                showReportDataProcessingInfo(message);
                
                if (allVehicles.length === 0) {
                    showNoMonitoringState();
                    return;
                }

                processInspectionData(response.data);
            } else {
                showNoDataState();
                showConfirmDialog({
                    title: '{{ __("Error") }}',
                    message: response.message || '{{ __("Unknown error occurred") }}',
                    confirmText: '{{ __("OK") }}',
                    cancelText: '',
                    type: 'danger',
                    onConfirm: function() {},
                    onCancel: function() {}
                });
            }
        })
        .catch(function(error) {
            hideLoadingState();
            showNoDataState();
            console.error('Error loading inspection data:', error);
            showConfirmDialog({
                title: '{{ __("Error") }}',
                message: '{{ __("Error loading inspection data") }}',
                confirmText: '{{ __("OK") }}',
                cancelText: '',
                type: 'danger',
                onConfirm: function() {},
                onCancel: function() {}
            });
        });
    }

    // Process inspection data
    function processInspectionData(data) {
        if (!data.vehicles || data.vehicles.length === 0) {
            showNoDataState();
            return;
        }

        // Separate vehicles with and without inspection monitoring
        vehiclesWithInspection = [];
        vehiclesWithoutInspection = [];

        data.vehicles.forEach(vehicle => {
            let hasInspectionMonitoring = false;
            let inspectionDate = null;
            let alarmStatus = false;
            
            if (vehicle.alarm_settings) {
                try {
                    const settings = JSON.parse(vehicle.alarm_settings);
                    hasInspectionMonitoring = settings.status_id_301 === "1";
                    inspectionDate = settings.inspection_date || null;
                    alarmStatus = settings.status_id_301 === "1";
                } catch (e) {
                    console.error('Error parsing alarm settings for vehicle:', vehicle.did, e);
                }
            }

            if (hasInspectionMonitoring && inspectionDate) {
                // Vehicle has inspection monitoring and inspection date
                vehiclesWithInspection.push(vehicle);
            } else {
                // Vehicle doesn't have inspection monitoring or inspection date
                vehiclesWithoutInspection.push(vehicle);
            }
        });

        // Process vehicles with inspection data
        const processedVehicles = vehiclesWithInspection.map((vehicle, index) => {
            // Parse alarm settings JSON
            let inspectionDate = null;
            let alarmStatus = false;
            
            if (vehicle.alarm_settings) {
                try {
                    const settings = JSON.parse(vehicle.alarm_settings);
                    inspectionDate = settings.inspection_date || null;
                    alarmStatus = settings.status_id_301 === "1";
                } catch (e) {
                    console.error('Error parsing alarm settings for vehicle:', vehicle.did, e);
                }
            }
            
            const today = new Date();
            const dueDate = inspectionDate ? new Date(inspectionDate) : null;
            
            let status = 'compliant';
            let daysRemaining = 0;
            
            if (dueDate) {
                const timeDiff = dueDate.getTime() - today.getTime();
                daysRemaining = Math.ceil(timeDiff / (1000 * 3600 * 24));
                
                if (daysRemaining < 0) {
                    status = 'overdue';
                } else if (daysRemaining <= 30) {
                    status = 'due_soon';
                } else if (daysRemaining <= 90) {
                    status = 'upcoming';
                } else {
                    status = 'compliant';
                }
            }
            
            return {
                id: index + 1,
                vehicle_id: vehicle.did,
                vehicle_name: vehicle.name || '{{ __("Unknown Vehicle") }}',
                plate: vehicle.tx?.plt || '',
                category_name: vehicle.category_name || '',
                inspection_date: inspectionDate,
                status: status,
                days_remaining: daysRemaining,
                alarm_status: alarmStatus,
                category_id: vehicle.category_id
            };
        });

        // Filter by status if specified
        const statusFilter = $('#statusFilter').val();
        let filteredVehicles = processedVehicles;
        
        if (statusFilter) {
            filteredVehicles = processedVehicles.filter(vehicle => vehicle.status === statusFilter);
        }



        if (filteredVehicles.length === 0) {
            showNoDataState();
            return;
        }

        // Calculate summary statistics
        const summary = calculateSummary(processedVehicles);
        
        // Display the processed data
        displayReport({
            summary: summary,
            detailed_data: filteredVehicles
        });
    }

    // Calculate summary statistics
    function calculateSummary(vehicles) {
        const totalVehicles = allVehicles.length;
        const withInspectionDate = vehiclesWithInspection.length;
        const overdueCount = vehicles.filter(v => v.status === 'overdue').length;
        const dueSoonCount = vehicles.filter(v => v.status === 'due_soon').length;
        const compliantCount = vehicles.filter(v => v.status === 'compliant').length;
        const upcomingCount = vehicles.filter(v => v.status === 'upcoming').length;
        
        return {
            total_vehicles: totalVehicles,
            with_inspection_date: withInspectionDate,
            overdue_count: overdueCount,
            due_soon_count: dueSoonCount,
            compliant_count: compliantCount,
            upcoming_count: upcomingCount
        };
    }

    // Display report data
    function displayReport(data) {
        // Show all sections
        $('#summaryCards').removeClass('d-none');
        $('#chartsRow').removeClass('d-none');
        $('#inspectionTableCard').removeClass('d-none');
        $('#noDataState').addClass('d-none');
        $('#noMonitoringState').addClass('d-none');

        // Update summary cards
        $('#totalVehicles').text(data.summary.total_vehicles);
        $('#withInspectionDate').text(data.summary.with_inspection_date);
        $('#overdueCount').text(data.summary.overdue_count);
        $('#dueSoonCount').text(data.summary.due_soon_count);
        $('#compliantCount').text(data.summary.compliant_count);

        // Display detailed data
        displayDetailedData(data.detailed_data);

        // Display vehicles without inspection date
        displayNoInspectionData();

        // Update charts
        updateCharts(data.detailed_data);
    }

    // Display detailed data table
    function displayDetailedData(data) {
        if (data && data.length > 0) {
            tableInspection.setData(data);
        } else {
            tableInspection.setData([]);
        }
    }

    // Display vehicles without inspection date
    function displayNoInspectionData() {
        if (vehiclesWithoutInspection.length === 0) {
            $('#noInspectionTableCard').addClass('d-none');
            return;
        }

        const processedNoInspection = vehiclesWithoutInspection.map((vehicle, index) => {
            let inspectionMonitoring = false;
            
            if (vehicle.alarm_settings) {
                try {
                    const settings = JSON.parse(vehicle.alarm_settings);
                    inspectionMonitoring = settings.status_id_301 === "1";
                } catch (e) {
                    console.error('Error parsing alarm settings for vehicle:', vehicle.did, e);
                }
            }

            return {
                id: index + 1,
                vehicle_id: vehicle.did,
                vehicle_name: vehicle.name || '{{ __("Unknown Vehicle") }}',
                plate: vehicle.tx?.plt || '',
                category_name: vehicle.category_name || '',
                inspection_monitoring: inspectionMonitoring,
                category_id: vehicle.category_id
            };
        });

        tableNoInspection.setData(processedNoInspection);
        $('#noInspectionTableCard').removeClass('d-none');
    }

    // Update charts
    function updateCharts(data) {
        // Status distribution chart
        const statusCounts = {
            overdue: data.filter(v => v.status === 'overdue').length,
            due_soon: data.filter(v => v.status === 'due_soon').length,
            upcoming: data.filter(v => v.status === 'upcoming').length,
            compliant: data.filter(v => v.status === 'compliant').length
        };

        renderStatusDistributionChart(statusCounts);

        // Monthly due dates chart
        const monthlyData = generateMonthlyData(data);
        renderMonthlyDueChart(monthlyData);
    }

    // Generate monthly data for chart
    function generateMonthlyData(data) {
        const monthlyCounts = {};
        
        data.forEach(vehicle => {
            if (vehicle.inspection_date) {
                const date = new Date(vehicle.inspection_date);
                const monthKey = `${date.getFullYear()}-${String(date.getMonth() + 1).padStart(2, '0')}`;
                monthlyCounts[monthKey] = (monthlyCounts[monthKey] || 0) + 1;
            }
        });

        return monthlyCounts;
    }

    // Render status distribution chart
    function renderStatusDistributionChart(statusCounts) {
        // Clear existing chart
        $("#statusDistributionChart").html('');
        
        const labels = [
            '{{ __("Overdue") }}',
            '{{ __("Due Soon") }}',
            '{{ __("Upcoming") }}',
            '{{ __("Compliant") }}'
        ];
        
        const data = [
            statusCounts.overdue,
            statusCounts.due_soon,
            statusCounts.upcoming,
            statusCounts.compliant
        ];
        
        const colors = ['#ef4444', '#f59e0b', '#3b82f6', '#10b981'];
        
        var options = {
            chart: {
                type: 'donut',
                height: 300,
                toolbar: { show: false }
            },
            series: data,
            labels: labels,
            colors: colors,
            legend: { position: 'bottom' },
            plotOptions: {
                pie: {
                    donut: {
                        size: '60%'
                    }
                }
            },
            dataLabels: {
                enabled: true,
                formatter: function(val, opts) {
                    return opts.w.globals.series[opts.seriesIndex];
                }
            }
        };
        
        new ApexCharts(document.querySelector("#statusDistributionChart"), options).render();
    }

    // Render monthly due chart
    function renderMonthlyDueChart(monthlyData) {
        // Clear existing chart
        $("#monthlyDueChart").html('');
        
        const months = Object.keys(monthlyData).sort();
        const counts = months.map(month => monthlyData[month]);
        
        var options = {
            chart: {
                type: 'bar',
                height: 300,
                toolbar: { show: false }
            },
            series: [
                { name: '{{ __("Due Inspections") }}', data: counts }
            ],
            xaxis: {
                categories: months.map(month => {
                    const [year, monthNum] = month.split('-');
                    return `${monthNum}/${year}`;
                }),
                title: { text: '{{ __("Month") }}' }
            },
            yaxis: {
                title: { text: '{{ __("Count") }}' },
                beginAtZero: true
            },
            colors: ['#3b82f6'],
            plotOptions: {
                bar: {
                    horizontal: false,
                    columnWidth: '55%',
                    endingShape: 'rounded'
                }
            },
            dataLabels: {
                enabled: true,
                formatter: function(val) {
                    return val;
                }
            }
        };
        
        new ApexCharts(document.querySelector("#monthlyDueChart"), options).render();
    }

    // Generate report (called from filter button)
    function generateReport() {
        loadInspectionData();
    }



    // Print report
    function printReport() {
        window.print();
    }

    // Toggle columns
    function toggleColumns() {
        showConfirmDialog({
            title: '{{ __("Info") }}',
            message: '{{ __("Column selector feature will be added") }}',
            confirmText: '{{ __("OK") }}',
            cancelText: '',
            type: 'info',
            onConfirm: function() {},
            onCancel: function() {}
        });
    }

    // State management functions
    function showLoadingState() {
        $('#loadingState').removeClass('d-none');
        $('#summaryCards').addClass('d-none');
        $('#chartsRow').addClass('d-none');
        $('#inspectionTableCard').addClass('d-none');
        $('#noInspectionTableCard').addClass('d-none');
        $('#noDataState').addClass('d-none');
        $('#noMonitoringState').addClass('d-none');
    }

    function hideLoadingState() {
        $('#loadingState').addClass('d-none');
    }

    function showNoDataState() {
        $('#noDataState').removeClass('d-none');
        $('#summaryCards').addClass('d-none');
        $('#chartsRow').addClass('d-none');
        $('#inspectionTableCard').addClass('d-none');
        $('#noInspectionTableCard').addClass('d-none');
        $('#noMonitoringState').addClass('d-none');
    }

    function showNoMonitoringState() {
        $('#noMonitoringState').removeClass('d-none');
        $('#summaryCards').addClass('d-none');
        $('#chartsRow').addClass('d-none');
        $('#inspectionTableCard').addClass('d-none');
        $('#noInspectionTableCard').addClass('d-none');
        $('#noDataState').addClass('d-none');
    }
</script>
@endsection
