@php
use App\Helpers\Base64Helper;

/* URL:
vehicle-map/eyJkaWQiOiJhZWJjYzJiMzc1YTI0ZjNmMCIsImRhdGUiOiIyMDI1LTA2LTIzIiwic3RhcnRfdGltZSI6IjAwOjAxIiwiZW5kX3RpbWUiOiIxMjozNiJ9
*/

$get_data = null;
if($data != "all"){
    $get_data = Base64Helper::decode($data);
}
//print_r($get_data);
$did = $get_data != null ? $get_data['did'] : '';
$date = $get_data != null ? $get_data['date'] : date('Y-m-d');
$start_date = $get_data != null ? $get_data['date'] : date('Y-m-d', strtotime('-7 days'));
$end_date = $get_data != null ? $get_data['date'] : date('Y-m-d');

@endphp

@extends('layouts.app')

@section('title', __('Daily Movement Summary') )

@section('content')

<!-- Page header -->
<div class="page-header d-print-none">
    <div class="container-xl">
        <div class="row g-2 align-items-center">
            <div class="col">
                <div class="page-pretitle">
                    {{ __('Analysis and Statistics') }}
                </div>
                <h2 class="page-title">
                    {{ __('Daily Movement Summary') }}
                </h2>
            </div>
            <div class="col-auto ms-auto d-print-none">
                <div class="btn-list">
                    <button type="button" class="btn btn-primary" onclick="window.print()">
                        <i class="ti ti-printer me-1"></i>
                        {{ __('Print Report') }}
                    </button>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Page body -->
<div class="page-body">
    <div class="container-xl">
        <!-- Filters -->
        <div class="row mb-3">
            <div class="col-12">
                <div class="card">
                    <div class="card-header">
                        <h3 class="card-title">
                            <i class="ti ti-filter me-2"></i>
                            {{ __('Filters') }}
                        </h3>
                    </div>
                    <div class="card-body">
                        <div class="row g-3">
                            <div class="col-md-3">
                                <label class="form-label">{{ __('Date Range') }}</label>
                                <div class="input-group">
                                    <input type="date" class="form-control" id="startDate" value="{{ $start_date }}">
                                    <span class="input-group-text">{{ __('to') }}</span>
                                    <input type="date" class="form-control" id="endDate" value="{{ $end_date }}">
                                </div>
                            </div>
                            <div class="col-md-3">
                                <label class="form-label">{{ __('Vehicle') }}</label>
                                <select class="form-select" id="vehicleFilter">
                                    <option value="">{{ __('All Vehicles') }}</option>
                                </select>
                            </div>
                            <div class="col-md-3">
                                <label class="form-label">{{ __('Category') }}</label>
                                <select class="form-select" id="categoryFilter">
                                    <option value="">{{ __('All Categories') }}</option>
                                </select>
                            </div>
                            <div class="col-md-3 d-flex align-items-end">
                                <div class="btn-group w-100">
                                    <button type="button" class="btn btn-primary" id="applyFilters">
                                        <i class="ti ti-search me-1"></i>
                                        {{ __('Apply Filters') }}
                                    </button>
                                    <button type="button" class="btn btn-secondary" id="resetFilters">
                                        <i class="ti ti-refresh me-1"></i>
                                        {{ __('Reset') }}
                                    </button>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Statistics Cards -->
        <div class="row row-deck row-cards mb-3" id="summaryCards">
            <div class="col-sm-6 col-lg-3">
                <div class="card">
                    <div class="card-body">
                        <div class="d-flex align-items-center">
                            <div class="subheader">{{ __('Total Distance') }}</div>
                        </div>
                        <div class="h1 mb-3" id="totalDistance">-</div>
                        <div class="d-flex mb-2">
                            <div>{{ __('Average per vehicle') }}</div>
                            <div class="ms-auto">
                                <span class="text-green d-inline-flex align-items-center lh-1">
                                    -
                                </span>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            <div class="col-sm-6 col-lg-3">
                <div class="card">
                    <div class="card-body">
                        <div class="d-flex align-items-center">
                            <div class="subheader">{{ __('Total Time') }}</div>
                        </div>
                        <div class="h1 mb-3" id="totalTime">-</div>
                        <div class="d-flex mb-2">
                            <div>{{ __('Average per vehicle') }}</div>
                            <div class="ms-auto">
                                <span class="text-green d-inline-flex align-items-center lh-1">
                                    -
                                </span>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            <div class="col-sm-6 col-lg-3">
                <div class="card">
                    <div class="card-body">
                        <div class="d-flex align-items-center">
                            <div class="subheader">{{ __('Active Vehicles') }}</div>
                        </div>
                        <div class="h1 mb-3" id="activeVehicles">-</div>
                        <div class="d-flex mb-2">
                            <div>{{ __('Total vehicles') }}</div>
                            <div class="ms-auto">
                                <span class="text-green d-inline-flex align-items-center lh-1">
                                    -
                                </span>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            <div class="col-sm-6 col-lg-3">
                <div class="card">
                    <div class="card-body">
                        <div class="d-flex align-items-center">
                            <div class="subheader">{{ __('Average Speed') }}</div>
                        </div>
                        <div class="h1 mb-3" id="avgSpeed">-</div>
                        <div class="d-flex mb-2">
                            <div>{{ __('Max speed') }}</div>
                            <div class="ms-auto">
                                <span class="text-red d-inline-flex align-items-center lh-1">
                                    -
                                </span>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Data Table -->
        <div class="row">
            <div class="col-12">
                <div class="card" id="dataTableCard">
                    <div class="card-header">
                        <h3 class="card-title">
                            <i class="ti ti-table me-2"></i>
                            {{ __('Daily Movement Data') }}
                        </h3>
                    </div>
                    <div class="card-body">
                        <div id="dailySummaryTable"></div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Loading State -->
        <div class="text-center py-4 d-none" id="loadingState">
            <div class="spinner-border text-primary" role="status">
                <span class="visually-hidden">{{ __('Loading...') }}</span>
            </div>
            <div class="mt-2">{{ __('Generating daily movement summary...') }}</div>
        </div>

        <!-- No Data State -->
        <div class="text-center py-4 d-none" id="noDataState">
            <div class="empty">
                <div class="empty-img">
                    <i class="ti ti-chart-line text-muted" style="font-size: 3rem;"></i>
                </div>
                <p class="empty-title">{{ __('No movement data found') }}</p>
                <p class="empty-subtitle text-muted">
                    {{ __('No movement data was found for the selected period and filters.') }}
                </p>
            </div>
        </div>
    </div>
</div>

<!-- Vehicle Details Modal -->
<div class="modal fade" id="vehicleDetailsModal" tabindex="-1" aria-labelledby="vehicleDetailsModalLabel"
    aria-hidden="true">
    <div class="modal-dialog modal-xl modal-dialog-centered">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="vehicleDetailsModalLabel">{{ __('Vehicle Details') }}</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="{{ __('Close') }}"></button>
            </div>
            <div class="modal-body">
                <div id="vehicleDetailsTable"></div>
            </div>
        </div>
    </div>
</div>
@endsection

@section('styles')
<style>
    .tabulator {
        font-size: 0.875rem;
    }

    .tabulator-header {
        background-color: #f8f9fa;
    }
</style>
@endsection

@section('scripts')
<script>

    var speedUnit = "";
    var lengthUnit = "";
    var currency = "";
    
    var get_data = Base64Helper.decode("{{ $data }}");
    get_data = get_data == null ? {} : get_data;

    // Global variables
    var tableDailySummary;
    var reportData = [];
    var vehicleList = [];
    var categoriesDevicesMatch = [];

    // Initialize page states
    $('#summaryCards').addClass('d-none');
    $('#dataTableCard').addClass('d-none');
    $('#loadingState').removeClass('d-none');
    $('#noDataState').addClass('d-none');

    onAppReady(function() {     
        initializePage();
    });
    
    function initializePage() {
        loadVehicles()
        .then(loadCategories)
        .then(loadCategoriesDevicesMatch)
        .then(initListeners)
        .then(initTabulator)
        .then(loadData);
    }
    
    function initListeners(){

        // Listen for vehicle/category filter changes to filter client-side
        $("#vehicleFilter, #categoryFilter").on("change", function() {
            applyClientFilters();
        });
        
        // Apply filters
        $("#applyFilters").on("click", function() {
            loadData();
        });

        // Reset filters
        $("#resetFilters").on("click", function() {
            $("#startDate").val("{{ date('Y-m-d', strtotime('-7 days')) }}");
            $("#endDate").val("{{ date('Y-m-d') }}");
            $("#vehicleFilter").val("");
            $("#categoryFilter").val("");
            //loadData();
        });
    }

    // Initialize Tabulator
    function initTabulator() {
        tableDailySummary = createTabulator("#dailySummaryTable", {
            columns: [
                {
                    title: "{{ __('Vehicle') }}",
                    field: "vehicle_name",
                    headerFilter: true,
                    sorter: "string"
                },
                {
                    title: "{{ __('Date') }}",
                    field: "date_ui",
                    headerFilter: true,
                    sorter: "string",
                    formatter: function(cell) {
                        // Use the _ui field for display
                        return cell.getRow().getData().date_ui || cell.getValue();
                    }
                },
                {
                    title: "{{ __('Distance') }}",
                    field: "distance",
                    headerFilter: true,
                    sorter: "number",
                    formatter: function(cell) {
                        return cell.getValue().toFixed(2) + " " + lengthUnit;
                    }
                },
                {
                    title: "{{ __('Duration') }}",
                    field: "duration",
                    headerFilter: true,
                    sorter: "number",
                    formatter: function(cell) {
                        var hours = Math.floor(cell.getValue() / 60);
                        var minutes = cell.getValue() % 60;
                        return hours + "h " + minutes + "m";
                    }
                },
                {
                    title: "{{ __('Avg Speed') }}",
                    field: "avg_speed",
                    headerFilter: true,
                    sorter: "number",
                    formatter: function(cell) {
                        return cell.getValue().toFixed(1) + " " + speedUnit;
                    }
                },
                {
                    title: "{{ __('Max Speed') }}",
                    field: "max_speed",
                    headerFilter: true,
                    sorter: "number",
                    formatter: function(cell) {
                        return cell.getValue().toFixed(1) + " " + speedUnit;
                    }
                },
                {
                    title: "{{ __('Actions') }}",
                    formatter: function(cell) {
                        var data = cell.getRow().getData();
                        var urlEncoded = Base64Helper.encode({
                            did: data.device_id,
                            date: data.date,
                            start_time: "00:01",
                            end_time: "23:59"
                        });
                        return '<a href="/vehicle-map/' + urlEncoded + '" class="btn btn-sm btn-primary me-2">' +
                               '<i class="ti ti-eye"></i> {{ __("Map") }}</a>' +
                               '<a onclick="viewDetails(\'' + data.device_id + '\', \'' + data.date + '\')" class="btn btn-sm btn-primary">' +
                               '<i class="ti ti-list-details"></i> {{ __("Details") }}</a>';
                    }
                }
            ]
        });
    }

    // Load vehicles
    async function loadVehicles() {
        return http.post({
            url: "/api/vehicles/list",
            loading: false
        }).then((response) => {
            if(response.success && response.data && response.data.vehicles){
                vehicleList = response.data.vehicles || [];

                var options = '<option value="">{{ __("All Vehicles") }}</option>';
                vehicleList.forEach(function(vehicle) {
                    options += '<option value="' + vehicle.did + '" ' + (get_data.did == vehicle.did ? 'selected' : '') + '>' + vehicle.name + '</option>';
                });
                $("#vehicleFilter").html(options);
            }
        }).catch(function(error) {
            console.error('{{ __("Error loading vehicles") }}:', error);
            showConfirmDialog({
                title: '{{ __("Error") }}',
                message: '{{ __("Failed to load vehicles") }}',
                confirmText: '{{ __("OK") }}',
                showCancelButton: false,
                type: 'danger',
                onConfirm: function() {},
                onCancel: function() {}
            });
        });
    }

    // Load categories
    async function loadCategories() {
        return http.post({
            url: "/categories-list",
            loading: false
        }).then((response) => {
            if(response.success && response.data){
                var options = '<option value="">{{ __("All Categories") }}</option>';
                response.data.forEach(function(category) {
                    options += '<option value="' + category.id + '">' + category.name + '</option>';
                });
                $("#categoryFilter").html(options); 
            }
        }).catch(function(error) {
            console.error('{{ __("Error loading categories") }}:', error);
        });
    }

    async function loadCategoriesDevicesMatch() {
        return http.post({
            url: "/api/categories-devices-match",
            loading: false
        }).then((response) => {
            if(response.success && response.data){
                categoriesDevicesMatch = response.data;
            }
        }).catch(function(error) {
            console.error('{{ __("Error loading categories devices match") }}:', error);
        });
    }

    // Load data
    async function loadData() {
        const startDate = $("#startDate").val();
        const endDate = $("#endDate").val();

        if (!startDate || !endDate) {
            showConfirmDialog({
                title: '{{ __("Warning") }}',
                message: '{{ __("Please select start and end dates") }}',
                confirmText: '{{ __("OK") }}',
                showCancelButton: false,
                type: 'warning',
                onConfirm: function() {},
                onCancel: function() {}
            });
            return;
        }

        // Show loading state
        showLoadingState();

        http.post({
            url: "/api/reports/daily-summary",
            data: {
                start_date: startDate,
                end_date: endDate,
                vehicle: $("#vehicleFilter").val(),
                category: $("#categoryFilter").val()
            },
            loading: false
        }).then((response) => {
            hideLoadingState();
            
            if (response.success) {
                // Set units from session settings
                speedUnit = __(window.sessionSettings.speedUnit || 'km/h');
                lengthUnit = __(window.sessionSettings.lengthUnit || 'km');
                currency = __(window.sessionSettings.currency || 'TRY');
                
                reportData = response.data.logs;

                console.log('reportData', reportData);

                var message = __('#name# used #value# records').replace('#name#', __('Daily Movement Summary')).replace('#value#', reportData.length);
                showReportDataProcessingInfo(message);

                processRawData(response.data.logs, function(processed) {
                    if (processed && processed.length > 0) {
                        displayReport({
                            summary: processed,
                            rawData: response.data.logs
                        });
                    } else {
                        showNoDataState();
                    }
                });
            } else {
                showNoDataState();
                showConfirmDialog({
                    title: '{{ __("Error") }}',
                    message: response.message || '{{ __("Unknown error occurred") }}',
                    confirmText: '{{ __("OK") }}',
                    showCancelButton: false,
                    type: 'danger',
                    onConfirm: function() {},
                    onCancel: function() {}
                });
            }
        }).catch(function(error) {
            hideLoadingState();
            showNoDataState();
            showConfirmDialog({
                title: '{{ __("Error") }}',
                message: __(error.message || '{{ __("Failed to load report data") }}'),
                confirmText: '{{ __("OK") }}',
                showCancelButton: false,
                type: 'danger',
                onConfirm: function() {},
                onCancel: function() {}
            });
        });
    }

    // Display report
    function displayReport(data) {
        // Show all sections
        $('#summaryCards').removeClass('d-none');
        $('#dataTableCard').removeClass('d-none');
        $('#noDataState').addClass('d-none');

        setTimeout(() => {
            // Update table
            tableDailySummary.setData(data.summary);
        }, calculateTimeRange(data.summary)); // more data more time
        
        // Update statistics
        updateStatistics(data.rawData);
    }

    // Process raw data into daily summary
    function processRawData(rawData, callback) {
        var vehicleData = {};
        var dailySummary = [];

        // Group data by device_id and date
        rawData.forEach(function(record) {
            if (!record.device_id || !record.timestamp) return;

            var date = record.timestamp.split(' ')[0];
            var key = record.device_id + '_' + date;

            if (!vehicleData[key]) {
                vehicleData[key] = {
                    device_id: record.device_id,
                    vehicle_name: getVehicleName(record.device_id),
                    date: date,
                    date_ui: record.timestamp_ui ? record.timestamp_ui.split(' ')[0] : date,
                    points: [],
                    total_distance: 0,
                    total_duration: 0,
                    speeds: [],
                    status_codes: []
                };
            }

            vehicleData[key].points.push({
                lat: parseFloat(record.latitude) || 0,
                lng: parseFloat(record.longitude) || 0,
                time: new Date(record.timestamp),
                log_data: record.log_data
            });
        });

        // Calculate statistics for each vehicle/day
        Object.keys(vehicleData).forEach(function(key) {
            var data = vehicleData[key];
            var points = data.points;

            if (points.length === 0) return;

            // Sort points by time
            points.sort(function(a, b) {
                return a.time - b.time;
            });

            // Calculate distance
            var totalDistance = 0;
            for (var i = 1; i < points.length; i++) {
                var distance = calculateDistance(
                    points[i-1].lat, points[i-1].lng,
                    points[i].lat, points[i].lng
                );
                
                totalDistance += distance;
            }

            // Calculate duration
            var duration = (points[points.length - 1].time - points[0].time) / (1000 * 60); // minutes

            // Extract speeds and status from log data
            var speeds = [];
            var statusCodes = [];
            points.forEach(function(point) {
                if (point.log_data) {
                    if (typeof point.log_data === "string") {
                        try { point.log_data = JSON.parse(point.log_data); } catch(e) {}
                    }
                    if (point.log_data.hiz !== undefined) {
                        speeds.push(parseFloat(point.log_data.hiz) || 0);
                    }
                    if (point.log_data.drx !== undefined) {
                        statusCodes.push(point.log_data.drx);
                    }
                }
            });

            // Calculate averages
            var avgSpeed = speeds.length > 0 ? speeds.reduce((a, b) => a + b, 0) / speeds.length : 0;
            var maxSpeed = speeds.length > 0 ? Math.max(...speeds) : 0;
            var lastStatus = statusCodes.length > 0 ? statusCodes[statusCodes.length - 1] : '0';

            // Only include if there's actual movement or duration
            if (totalDistance > 0 || duration > 0) {
                var statusInfo = getStatusInfo(lastStatus);
                dailySummary.push({
                    device_id: data.device_id,
                    vehicle_name: data.vehicle_name,
                    date: data.date,
                    date_ui: data.date_ui,
                    distance: Math.round(totalDistance * 100) / 100,
                    duration: Math.round(duration),
                    avg_speed: Math.round(avgSpeed * 10) / 10,
                    max_speed: Math.round(maxSpeed * 10) / 10,
                    status: statusInfo.text,
                    status_color: statusInfo.color
                });
            }
        });

        if (callback) callback(dailySummary);
    }

    // Convert status code to text and color
    function getStatusInfo(statusCode) {
        switch(statusCode) {
            case 0: 
                return { text: 'Stopping', color: '#dc3545', icon: 'ti ti-player-stop' }; // Red
            case 1: 
                return { text: 'Waiting', color: '#ffc107', icon: 'ti ti-player-pause' }; // Yellow
            case 2: 
                return { text: 'Moving', color: '#28a745', icon: 'ti ti-player-play' }; // Green
            case 3: 
                return { text: 'Pulling', color: '#000000', icon: 'ti ti-skip-forward' }; // Black
            case 4: 
                return { text: 'No Signal', color: '#6c757d', icon: 'ti ti-playlist-off' }; // Gray
            default: 
                return { text: 'Unknown', color: '#000000', icon: 'ti ti-pacman' }; // Red
        }
    }

    // Helper to get vehicle name by device_id
    function getVehicleName(device_id) {
        if (vehicleList) {
            var v = vehicleList.find(function(v) { return v.did == device_id; });
            return v ? v.name : device_id;
        }
        return device_id;
    }

    // Calculate and update statistics
    function updateStatistics(rawData) {
        processRawData(rawData, function(processedData) {
            var totalDistance = 0;
            var totalDuration = 0;
            var allSpeeds = [];
            var allMaxSpeeds = [];
            var vehicleCount = new Set();
            var totalVehicles = 0;
            
            processedData.forEach(function(record) {
                totalDistance += record.distance;
                totalDuration += record.duration;
                allSpeeds.push(record.avg_speed);
                allMaxSpeeds.push(record.max_speed);
                vehicleCount.add(record.device_id);
            });
            
            // Get total number of vehicles from the filter dropdown
            totalVehicles = $("#vehicleFilter option").length - 1; // Subtract 1 for "All Vehicles" option
            
            var avgSpeed = allSpeeds.length > 0 ? allSpeeds.reduce((a, b) => a + b, 0) / allSpeeds.length : 0;
            var maxSpeed = allMaxSpeeds.length > 0 ? Math.max(...allMaxSpeeds) : 0;
            
            // Calculate averages per vehicle
            var avgDistancePerVehicle = vehicleCount.size > 0 ? totalDistance / vehicleCount.size : 0;
            var avgTimePerVehicle = vehicleCount.size > 0 ? totalDuration / vehicleCount.size : 0;
            
            $("#totalDistance").text(totalDistance.toFixed(2) + " " + lengthUnit);
            $("#totalTime").text(formatDuration(totalDuration));
            $("#activeVehicles").text(vehicleCount.size);
            $("#avgSpeed").text(avgSpeed.toFixed(1) + " " + speedUnit);
            
            // Update averages in ms-auto sections
            $(".card:has(#totalDistance) .ms-auto .text-green").text(avgDistancePerVehicle.toFixed(2) + " " + lengthUnit);
            $(".card:has(#totalTime) .ms-auto .text-green").text(formatDuration(avgTimePerVehicle.toFixed(0)));
            $(".card:has(#activeVehicles) .ms-auto .text-green").text(totalVehicles.toFixed(0));
            
            // Update max speed in the average speed card
            $(".card:has(#avgSpeed) .text-red").text(maxSpeed.toFixed(1) + " " + speedUnit);
        });
    }

    // Format duration
    function formatDuration(minutes) {
        var hours = Math.floor(minutes / 60);
        var mins = minutes % 60;
        return hours + __("h") + " " + mins + __("m");
    }

    // View details
    function viewDetails(deviceId, date) {
        // Show the modal
        $('#vehicleDetailsModal').modal('show');

        // Show loading spinner in the table area
        $("#vehicleDetailsTable").html('<div class="text-center p-4"><div class="spinner-border text-primary" role="status"><span class="visually-hidden">Loading...</span></div><div class="mt-2">{{ __("Loading...") }}</div></div>');

        // Filter the global raw data for the selected vehicle and date
        var detailsRawData = [];
        if (reportData) {
            detailsRawData = reportData.filter(function(item) {
                return item.device_id == deviceId && item.timestamp.split(' ')[0] == date;
            });
        }
        var details = [];
        detailsRawData.map(function(a, b) {
            var log_data = a.log_data;
            if (typeof log_data === "string") {
                try { log_data = JSON.parse(log_data); } catch(e) {}
            }
            details.push({
                timestamp_ui: a.timestamp_ui,
                hiz: log_data.hiz,
                drx: log_data.drx
            });
        });

        // Initialize Tabulator with the filtered data
        createTabulator("#vehicleDetailsTable", {
            data: details,
            columns: [
                { title: "{{ __('Time') }}", field: "timestamp_ui", formatter: function(cell) {
                    return cell.getValue();
                } },
                { title: "{{ __('Speed') }}", field: "hiz", formatter: function(cell) {
                    return cell.getValue() != undefined ? parseFloat(cell.getValue()).toFixed(1) + " " + speedUnit : "-";
                } },
                { title: "{{ __('Status') }}", field: "drx", formatter: function(cell) {
                    var status = getStatusInfo(cell.getValue());
                    return '<span class="badge text-white" style="background-color: ' + status.color + ';"> <i class="' + status.icon + '"></i> ' + __(status.text) + '</span>';
                } },
            ]
        });
    }

    // Apply client-side filters to reportData
    function applyClientFilters() {
        var selectedVehicle = $("#vehicleFilter").val();
        var selectedCategory = $("#categoryFilter").val();
        var filteredLogs = reportData || [];

        // Filter by category using categoriesDevicesMatch
        if (selectedCategory) {
            var allowedDeviceIds = (categoriesDevicesMatch || [])
                .filter(function(match) { return match.category_id == selectedCategory; })
                .map(function(match) { return match.device_id; });
            filteredLogs = filteredLogs.filter(function(log) {
                return allowedDeviceIds.includes(log.device_id);
            });
        }

        // Filter by vehicle using selected value
        if (selectedVehicle) {
            filteredLogs = filteredLogs.filter(function(log) {
                return log.device_id == selectedVehicle;
            });
        }

        processRawData(filteredLogs, function(processed) {
            if (processed && processed.length > 0) {
                tableDailySummary.setData(processed);
                updateStatistics(filteredLogs);
            } else {
                tableDailySummary.setData([]);
                // Reset statistics to zero
                $("#totalDistance").text("0 " + lengthUnit);
                $("#totalTime").text("0h 0m");
                $("#activeVehicles").text("0");
                $("#avgSpeed").text("0 " + speedUnit);
                $(".card:has(#totalDistance) .ms-auto .text-green").text("0 " + lengthUnit);
                $(".card:has(#totalTime) .ms-auto .text-green").text("0h 0m");
                $(".card:has(#activeVehicles) .ms-auto .text-green").text("0");
                $(".card:has(#avgSpeed) .text-red").text("0 " + speedUnit);
            }
        });
    }

    // Show loading state
    function showLoadingState() {
        $('#loadingState').removeClass('d-none');
        $('#summaryCards').addClass('d-none');
        $('#dataTableCard').addClass('d-none');
        $('#noDataState').addClass('d-none');
    }

    // Hide loading state
    function hideLoadingState() {
        $('#loadingState').addClass('d-none');
    }

    // Show no data state
    function showNoDataState() {
        $('#noDataState').removeClass('d-none');
        $('#summaryCards').addClass('d-none');
        $('#dataTableCard').addClass('d-none');
    }
</script>
@endsection