LeafletMap = function () {
    this.markerBounds = [];
    this.mapLet = null;
    this.markersAll = [];
    this.markersCluster = L.markerClusterGroup();
    this.divId = null;
    this.polygons = [];
    this.polygonMap = new Map(); // ID -> Polygon object mapping
    this.drawControl = null; // Drawing control reference
    this.drawnItems = null; // Layer group for drawn items

    this.initMap = function (divId, settings) {
        this.divId = divId;
        var zoomLevelDefault = settings.zoomLevelDefault || 20;

        // Initialize drawn items layer group
        this.drawnItems = new L.FeatureGroup();

        this.markersCluster = L.markerClusterGroup({
            chunkedLoading: settings.chunkedLoading || true,
            disableClusteringAtZoom: settings.disableClusteringAtZoom || 10,
            showHideDelay: settings.showHideDelay || 100,
            spiderfyOnMaxZoom: settings.spiderfyOnMaxZoom || false,
        });

        this.mapLet = L.map(divId).setView(
            [39.5815051, 34.5783959], // add Türkiye coordinates
            zoomLevelDefault
        );

        bounds = L.latLngBounds([39.5815051, 34.5783959], [41.0082, 28.9784]);
        this.mapLet.fitBounds(bounds);

        // Add drawn items to map
        this.mapLet.addLayer(this.drawnItems);

        // Initialize drawing controls if enabled
        if (settings.enablePolygonDrawing !== false) {
            this.initDrawingControls(settings.drawingOptions || {});
        }

        var osm = L.tileLayer(
            "https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png",
            {
                crs_name: "other",
                attribution: "© OpenStreetMap",
            }
        );
        var google_streets = L.tileLayer(
            "http://{s}.google.com/vt/lyrs=m&x={x}&y={y}&z={z}&lang=tr_TR",
            {
                crs_name: "other",
                maxZoom: zoomLevelDefault,
                subdomains: ["mt0", "mt1", "mt2", "mt3"],
                attribution: "Google",
            }
        );
        var google_hybrid = L.tileLayer(
            "http://{s}.google.com/vt/lyrs=s,h&x={x}&y={y}&z={z}&lang=tr_TR",
            {
                crs_name: "other",
                maxZoom: zoomLevelDefault,
                subdomains: ["mt0", "mt1", "mt2", "mt3"],
                attribution: "Google",
            }
        );
        var google_satellite = L.tileLayer(
            "http://{s}.google.com/vt/lyrs=s&x={x}&y={y}&z={z}&lang=tr_TR",
            {
                crs_name: "other",
                maxZoom: zoomLevelDefault,
                subdomains: ["mt0", "mt1", "mt2", "mt3"],
                attribution: "Google",
            }
        );
        var google_terrain = L.tileLayer(
            "http://{s}.google.com/vt/lyrs=p&x={x}&y={y}&z={z}&lang=tr_TR",
            {
                crs_name: "other",
                maxZoom: zoomLevelDefault,
                subdomains: ["mt0", "mt1", "mt2", "mt3"],
                attribution: "Google",
            }
        );
        var yandex_map = L.tileLayer(
            "https://core-renderer-tiles.maps.yandex.net/tiles?l=map&x={x}&y={y}&z={z}&scale=1&lang=tr_TR",
            {
                crs_name: "yandex",
                maxNativeZoom: zoomLevelDefault,
                attribution: "Yandex",
            }
        );
        var yandex_satellite = L.tileLayer(
            "https://core-sat.maps.yandex.net/tiles?l=sat&x={x}&y={y}&z={z}&scale=1&lang=tr_TR",
            {
                crs_name: "yandex",
                maxNativeZoom: zoomLevelDefault,
                attribution: "Yandex",
            }
        );
        var yandex_hybrid = L.tileLayer(
            "https://core-renderer-tiles.maps.yandex.net/tiles?l=skl&x={x}&y={y}&z={z}&scale=1&lang=tr_TR",
            {
                crs_name: "yandex",
                maxNativeZoom: zoomLevelDefault,
                attribution: "Yandex",
            }
        );
        var gis = L.tileLayer(
            "https://{s}.maps.2gis.com/tiles?x={x}&y={y}&z={z}&v=1",
            {
                crs_name: "other",
                maxNativeZoom: zoomLevelDefault,
                subdomains: ["tile0", "tile1", "tile2", "tile3"],
                attribution: "2GIS",
            }
        );
        var yandex_traffic = L.tileLayer(
            "https://{s}.maps.yandex.net/tiles?l=trf&v=21.06.18-0-b210520094930&x={x}&y={y}&z={z}&scale=1&lang=tr_TR",
            {
                crs_name: "yandex",
                maxNativeZoom: zoomLevelDefault,
                subdomains: ["trf01", "trf02", "trf03", "trf04"],
                attribution: "Yandex",
            }
        );
        var yandex_panorama = L.tileLayer(
            "https://{s}.maps.yandex.net/tiles?l=pmap&v=21.06.18-0-b210520094930&x={x}&y={y}&z={z}&scale=1&lang=tr_TR",
            {
                crs_name: "yandex",
                maxNativeZoom: zoomLevelDefault,
                subdomains: ["pmap01", "pmap02", "pmap03", "pmap04"],
                attribution: "Yandex",
            }
        );
        var stamen_toner = L.tileLayer(
            "https://{s}.tile.stamen.com/toner/{z}/{x}/{y}.png",
            {
                attribution: "Stamen Toner",
            }
        );

        google_streets.addTo(this.mapLet);

        var baseMaps = {
            OpenStreetMap: osm,
            "Google Harita": google_streets,
            "Google Uydu": google_satellite,
            "Google Hibrit": google_hybrid,
            "Google Arazi": google_terrain,
            "Yandex Harita": yandex_map,
            "Yandex Uydu": yandex_satellite,
            "Yandex Hibrit": yandex_hybrid,
            Gis: gis,
        };

        var overlayMaps = {};

        L.control.layers(baseMaps, overlayMaps).addTo(this.mapLet);
        var _this = this;
        this.mapLet.on("baselayerchange", function (e) {
            var center = _this.mapLet.getCenter();
            var zoom = _this.mapLet.getZoom();
            _this.mapLet.options.crs =
                e.layer.options.crs_name === "yandex"
                    ? L.CRS.EPSG3395
                    : L.CRS.EPSG3857;
            _this.mapLet._resetView(center, zoom, false, false);
        });

        this.canvasRenderer = L.canvas();
    };

    // Initialize drawing controls for polygon creation
    this.initDrawingControls = function (options) {
        if (options.enablePolygonDrawing !== true) {
            return;
        }
        // Check if Leaflet.draw is loaded
        if (typeof L.Control.Draw === "undefined") {
            console.warn(
                "Leaflet.draw plugin is not loaded. Drawing controls will not be available."
            );
            console.warn(
                'Please include: <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/leaflet.draw/1.0.4/leaflet.draw.css" />'
            );
            console.warn(
                'Please include: <script src="https://cdnjs.cloudflare.com/ajax/libs/leaflet.draw/1.0.4/leaflet.draw.js"></script>'
            );
            return;
        }

        var drawOptions = {
            position: options.position || "topleft",
            draw: {
                polygon:
                    options.enablePolygon !== false
                        ? {
                              allowIntersection:
                                  options.allowIntersection || false,
                              drawError: {
                                  color: "#e1e100",
                                  message:
                                      "<strong>Error:</strong> Shape edges cannot cross!",
                              },
                              shapeOptions: {
                                  color: options.polygonColor || "#d32f2f",
                                  weight: options.polygonWeight || 3,
                                  fillColor:
                                      options.polygonFillColor || "#ffcdd2",
                                  fillOpacity:
                                      options.polygonFillOpacity || 0.3,
                              },
                          }
                        : false,
                rectangle:
                    options.enableRectangle !== false
                        ? {
                              shapeOptions: {
                                  color: options.rectangleColor || "#d32f2f",
                                  weight: options.rectangleWeight || 3,
                                  fillColor:
                                      options.rectangleFillColor || "#ffcdd2",
                                  fillOpacity:
                                      options.rectangleFillOpacity || 0.3,
                              },
                          }
                        : false,
                circle: options.enableCircle || false,
                circlemarker: options.enableCircleMarker || false,
                marker: options.enableMarker || false,
                polyline: options.enablePolyline || false,
            },
            edit: {
                featureGroup: this.drawnItems,
                remove: options.enableDelete !== false,
            },
        };

        this.drawControl = new L.Control.Draw(drawOptions);
        this.mapLet.addControl(this.drawControl);

        // Set up event handlers for drawing
        this.setupDrawingEventHandlers();
    };

    // Setup event handlers for drawing operations
    this.setupDrawingEventHandlers = function () {
        // Check if Leaflet.draw is loaded
        if (typeof L.Draw === "undefined") {
            return;
        }

        var _this = this;

        this.mapLet.on(L.Draw.Event.CREATED, function (e) {
            var type = e.layerType;
            var layer = e.layer;

            if (type === "polygon" || type === "rectangle") {
                var polygonId = rand_id();
                layer.options.id = polygonId;

                // Add to drawn items
                _this.drawnItems.addLayer(layer);

                // Add to our polygon tracking
                _this.polygons.push(layer);
                _this.polygonMap.set(polygonId, layer);

                // Bind popup with ID
                layer.bindPopup(`Polygon ID: ${polygonId}`, {});

                // Trigger custom event if callback is provided
                if (_this.onPolygonCreated) {
                    _this.onPolygonCreated(layer, polygonId);
                }
            }
        });

        this.mapLet.on(L.Draw.Event.EDITED, function (e) {
            var layers = e.layers;
            layers.eachLayer(function (layer) {
                // Trigger custom event if callback is provided
                if (_this.onPolygonEdited) {
                    _this.onPolygonEdited(layer, layer.options.id);
                }
            });
        });

        this.mapLet.on(L.Draw.Event.DELETED, function (e) {
            var layers = e.layers;
            layers.eachLayer(function (layer) {
                var polygonId = layer.options.id;

                // Remove from our tracking
                _this.removePolygonFromTracking(polygonId);

                // Trigger custom event if callback is provided
                if (_this.onPolygonDeleted) {
                    _this.onPolygonDeleted(layer, polygonId);
                }
            });
        });
    };

    // Enable/disable drawing controls
    this.toggleDrawingControls = function (enable) {
        if (typeof L.Control.Draw === "undefined") {
            console.warn(
                "Leaflet.draw plugin is not loaded. Cannot toggle drawing controls."
            );
            return false;
        }

        if (enable && !this.drawControl) {
            this.initDrawingControls({});
        } else if (!enable && this.drawControl) {
            this.mapLet.removeControl(this.drawControl);
            this.drawControl = null;
        }
        return true;
    };

    // Set custom event callbacks
    this.setPolygonEventCallbacks = function (callbacks) {
        this.onPolygonCreated = callbacks.onCreated || null;
        this.onPolygonEdited = callbacks.onEdited || null;
        this.onPolygonDeleted = callbacks.onDeleted || null;
    };

    this.pinMarkers = function (markers) {
        var canvasRenderer = L.canvas();
        var _this = this;
        markers.forEach(function (m) {
            if (m.coord[0] == 0 && m.coord[1] == 0) {
                return;
            }

            var defaultColor = "black";
            var defaultSize = 24;
            m.icon.direction =
                m.icon.direction != null &&
                m.icon.direction !== undefined &&
                m.icon.direction > 0
                    ? m.icon.direction - 28
                    : 0;
            m.icon.width = m.icon.width || defaultSize;

            var iconColor =
                m.icon.color != null ? `color: ${m.icon.color};` : "";
            var iconSize = `font-size: ${m.icon.width}px !important;`;
            var iconRotate = `transform: rotate(${m.icon.direction}deg);`;
            var iconAnchor = m.icon.anchor || null;
            var icon = null;

            if (m.icon.name !== null) {
                icon = L.divIcon({
                    className: `${m.id} ${_this.divId}_marker 
                                ${
                                    m.markerClassName != null
                                        ? m.markerClassName
                                        : ""
                                }`,
                    html: `
                        <div class="map-icon">
                            <div class="map-icon-label
                                ${m.label == null ? "d-none" : ""}">
                                ${m.label}
                            </div>
                            <i class="${m.icon.name} map-icon-this
                                ${m.icon.className}"
                                id="${m.id}"
                                style="${iconColor} ${iconSize} ${iconRotate}">
                            </i>
                        </div>
                    `,
                    iconSize: [m.icon.width, m.icon.width],
                    iconAnchor: iconAnchor,
                });
            } else {
                var iconSize = `font-size: 12px !important;`;
                icon = L.divIcon({
                    className: `${m.id} ${_this.divId}_marker 
                    ${m.iconClassName != null ? m.iconClassName : ""}`,
                    html: `
                        <div class="map-icon">
                            <div class="map-icon-label
                                ${m.label == null ? "d-none" : ""}">
                                ${m.label}
                            </div>
                            <i class="ti ti-circle-filled map-icon-this
                                ${m.icon.className}"
                                style="${iconColor} ${iconSize} ${iconRotate}"></i>
                        </div>
                    `,
                    iconSize: [m.icon.width, m.icon.width],
                });
            }

            var marker = L.marker(m.coord, {
                id: m.id,
                icon: icon,
                renderer: canvasRenderer,
                draggable: m.draggable || false,
                riseOnHover: true,
                riseOffset: 1000,
            });

            if (m.draggable && typeof m.func_onDragEnd === "function") {
                marker.on("dragend", function (e) {
                    var newLatLng = e.target.getLatLng();
                    m.func_onDragEnd(m.id, newLatLng); // Use m.id as markerId
                });
            }

            if (m.popup != null) {
                marker.bindPopup(m.popup);
            }

            _this.markerBounds.push(marker.getLatLng());
            _this.markersCluster.addLayer(marker);
        });

        _this.markersAll.push(...markers);
        this.fitBounds();
        this.mapLet.addLayer(this.markersCluster);
    };

    this.fitBounds = function (markerIds) {
        if (markerIds && markerIds.length > 0) {
            // Fit bounds to visible markers
            const visibleLatLngs = [];
            myMap.markersCluster
                .getLayers()
                .filter((layer) => markerIds.includes(layer.options.id))
                .forEach((layer) => {
                    visibleLatLngs.push(layer.getLatLng());
                });

            if (visibleLatLngs.length > 0) {
                myMap.mapLet.fitBounds(visibleLatLngs);
            }
        } else {
            // Check if markerBounds has valid data
            if (this.markerBounds && this.markerBounds.length > 0) {
                try {
                    this.mapLet.fitBounds(this.markerBounds);
                } catch (e) {
                    console.warn("Could not fit bounds:", e);
                    // If only one marker, center on it instead
                    if (this.markerBounds.length === 1) {
                        this.mapLet.setView(this.markerBounds[0], 13);
                    }
                }
            }
        }
    };

    this.focusMarker = function (markerId) {
        var marker = this.markersCluster.getLayers().find(function (m) {
            return m.options.id === markerId;
        });

        if (marker) {
            this.mapLet.setView(marker.getLatLng(), 20);
            marker.openPopup();
        } else {
            console.warn(`Marker with ID '${markerId}' not found.`);
        }
    };

    this.refreshMap = async function () {
        await new Promise((resolve) => setTimeout(resolve, 100));
        this.mapLet.invalidateSize();
    };

    this.makeMarkerDraggable = function (markerId, draggable, func_onDragEnd) {
        var marker = this.markersCluster.getLayers().find(function (m) {
            return m.options.id === markerId;
        });

        if (marker) {
            // Check if dragging handler exists, if not initialize it
            if (!marker.dragging) {
                marker.options.draggable = true;
                if (marker._map) {
                    marker._map.removeLayer(marker);
                    marker._map.addLayer(marker);
                }
            }

            // Now enable or disable dragging
            if (marker.dragging) {
                marker.dragging[draggable ? "enable" : "disable"]();
            }

            marker.options.draggable = draggable;
            marker.setOpacity(draggable ? 1 : 0.5); // Optional visual feedback

            // Remove previous dragend event to avoid multiple bindings
            marker.off("dragend");

            if (draggable && typeof func_onDragEnd === "function") {
                marker.on("dragend", function (e) {
                    var newLatLng = e.target.getLatLng();
                    func_onDragEnd(markerId, newLatLng);
                });
            }
        } else {
            console.warn(`Marker with ID '${markerId}' not found.`);
        }
    };

    this.filterMarkers = function (model) {
        if (model.ids.length > 0) {
            model.ids.forEach(function (id) {
                $(`.${id}`).css("display", model.show ? "block" : "none");
            });
        } else {
            $(`.${this.divId}_marker`).css("opacity", model.show);
        }
    };

    this.addPolyline = function (polyline) {
        var polylineOptions = {
            color: polyline.color || "#000000",
            weight: polyline.weight || 3,
            opacity: polyline.opacity || 0.7,
            smoothFactor: polyline.smoothFactor || 1,
        };
        var line = L.polyline(polyline.coords, polylineOptions);
        line.addTo(this.mapLet);

        L.polylineDecorator(polyline.coords, {
            patterns: [
                {
                    offset: 25,
                    repeat: 50,
                    symbol: L.Symbol.arrowHead({
                        pixelSize: polyline.weight * 4,
                        pathOptions: {
                            fillOpacity: polyline.opacity,
                            weight: 0,
                            color: polyline.color,
                        },
                    }),
                },
            ],
        }).addTo(this.mapLet);
    };

    this.removePolyline = function () {
        var _this = this;
        this.mapLet.eachLayer(function (layer) {
            if (layer instanceof L.Polyline) {
                _this.mapLet.removeLayer(layer);
            }
        });
    };

    this.removeMarkers = function (markerIds) {
        var _this = this;
        if (Array.isArray(markerIds) || markerIds > 0) {
            if (!Array.isArray(markerIds)) {
                markerIds = [markerIds];
            }

            markerIds.forEach(function (id) {
                var marker = _this.markersCluster
                    .getLayers()
                    .find(function (m) {
                        return m.options.id === id;
                    });

                if (marker) {
                    _this.mapLet.removeLayer(marker);
                    _this.markersCluster.removeLayer(marker);
                    _this.markerBounds = _this.markerBounds.filter(function (
                        latlng
                    ) {
                        return !latlng.equals(marker.getLatLng());
                    });
                } else {
                    console.warn(`Marker with ID '${id}' not found.`);
                }
            });
            _this.markersAll = _this.markersAll.filter(function (m) {
                return !markerIds.includes(m.id);
            });
            if (_this.markerBounds.length === 0) {
                _this.mapLet.setView([41.0082, 28.9784], 20); // Reset view if no markers left
            } else {
                _this.fitBounds();
            }
        } else {
            // Remove all markers
            _this.markersCluster.clearLayers();
            _this.markerBounds = [];
            _this.markersAll = [];
            _this.mapLet.setView([41.0082, 28.9784], 20); // Reset view
        }
    };

    // Add a polygon programmatically
    this.addPolygon = function (polygon) {
        if (!polygon.coords || polygon.coords.length === 0) {
            console.warn("Polygon coordinates are empty.");
            return null;
        }
        if (polygon.coords.length < 3) {
            console.warn("Polygon coordinates must have at least 3 points.");
            return null;
        }

        if (polygon.coords[0].lat && polygon.coords[0].lng) {
            polygon.coords = polygon.coords.map(function (coord) {
                return [coord.lat, coord.lng];
            });
        }

        var polygonId = polygon.id || rand_id();
        var polygonOptions = {
            color: polygon.color || "#d32f2f",
            opacity: polygon.opacity || 0.7,
            weight: polygon.weight || 3,
            fillColor: polygon.fillColor || "#ffcdd2",
            fillOpacity: polygon.fillOpacity || 0.3,
            dashArray: polygon.dashArray || null,
            dashOffset: polygon.dashOffset || null,
            lineCap: polygon.lineCap || "round",
            lineJoin: polygon.lineJoin || "round",
            noClip: polygon.noClip || false,
            interactive:
                polygon.interactive !== undefined ? polygon.interactive : true,
            bubblingMouseEvents:
                polygon.bubblingMouseEvents !== undefined
                    ? polygon.bubblingMouseEvents
                    : true,
            id: polygonId,
        };

        var poly = L.polygon(polygon.coords, polygonOptions);
        poly.addTo(this.mapLet);
        poly.bindPopup(
            polygon.label || `Polygon ID: ${polygonId}`,
            polygon.popupOptions || {}
        );

        // Add to tracking arrays
        this.polygons.push(poly);
        this.polygonMap.set(polygonId, poly);

        // Add to drawn items if it exists
        if (this.drawnItems) {
            this.drawnItems.addLayer(poly);
        }

        return polygonId;
    };

    // Fit bounds to specific polygon(s) by ID
    this.fitBoundsToPolygons = function (polygonIds) {
        var polygons = [];

        if (!polygonIds) {
            // If no IDs provided, fit to all polygons
            polygons = this.polygons;
        } else if (typeof polygonIds === "string") {
            // Single polygon ID
            var polygon = this.polygonMap.get(polygonIds);
            if (polygon) {
                polygons = [polygon];
            }
        } else if (Array.isArray(polygonIds)) {
            // Multiple polygon IDs
            var _this = this;
            polygonIds.forEach(function (id) {
                var polygon = _this.polygonMap.get(id);
                if (polygon) {
                    polygons.push(polygon);
                }
            });
        }

        if (polygons.length === 0) {
            console.warn("No polygons found to fit bounds to.");
            return false;
        }

        var group = new L.featureGroup(polygons);
        this.mapLet.fitBounds(group.getBounds(), {
            padding: [10, 10],
        });
        return true;
    };

    // Remove polygon by ID
    this.removePolygon = function (polygonId) {
        var polygon = this.polygonMap.get(polygonId);
        if (!polygon) {
            console.warn(`Polygon with ID '${polygonId}' not found.`);
            return false;
        }

        // Remove from map
        this.mapLet.removeLayer(polygon);

        // Remove from drawn items
        if (this.drawnItems) {
            this.drawnItems.removeLayer(polygon);
        }

        // Remove from tracking
        this.removePolygonFromTracking(polygonId);
        return true;
    };

    // Remove polygon from internal tracking
    this.removePolygonFromTracking = function (polygonId) {
        // Remove from Map
        this.polygonMap.delete(polygonId);

        // Remove from array
        this.polygons = this.polygons.filter(function (poly) {
            return poly.options.id !== polygonId;
        });
    };

    // Get polygon by ID
    this.getPolygon = function (polygonId) {
        return this.polygonMap.get(polygonId) || null;
    };

    // Get all polygon IDs
    this.getPolygonIds = function () {
        return Array.from(this.polygonMap.keys());
    };

    // Check if polygon exists
    this.hasPolygon = function (polygonId) {
        return this.polygonMap.has(polygonId);
    };

    // Clear all polygons
    this.clearAllPolygons = function () {
        var _this = this;
        this.polygons.forEach(function (polygon) {
            _this.mapLet.removeLayer(polygon);
            if (_this.drawnItems) {
                _this.drawnItems.removeLayer(polygon);
            }
        });

        this.polygons = [];
        this.polygonMap.clear();
    };

    // Update polygon style
    this.updatePolygonStyle = function (polygonId, newStyle) {
        var polygon = this.polygonMap.get(polygonId);
        if (!polygon) {
            console.warn(`Polygon with ID '${polygonId}' not found.`);
            return false;
        }

        polygon.setStyle(newStyle);
        return true;
    };

    // Get polygon coordinates
    this.getPolygonCoordinates = function (polygonId) {
        var polygon = this.polygonMap.get(polygonId);
        if (!polygon) {
            console.warn(`Polygon with ID '${polygonId}' not found.`);
            return null;
        }

        return polygon.getLatLngs();
    };

    // Export polygon data
    this.exportPolygonData = function (polygonId) {
        if (polygonId) {
            var polygon = this.polygonMap.get(polygonId);
            if (!polygon) {
                console.warn(`Polygon with ID '${polygonId}' not found.`);
                return null;
            }

            return {
                id: polygon.options.id,
                coordinates: polygon.getLatLngs(),
                style: polygon.options,
            };
        } else {
            // Export all polygons
            var exportData = [];
            this.polygonMap.forEach(function (polygon, id) {
                exportData.push({
                    id: id,
                    coordinates: polygon.getLatLngs(),
                    style: polygon.options,
                });
            });
            return exportData;
        }
    };

    // Import polygon data
    this.importPolygonData = function (polygonDataArray) {
        var _this = this;
        var importedIds = [];

        polygonDataArray.forEach(function (polygonData) {
            var coords = polygonData.coordinates;
            if (coords && coords.length > 0) {
                // Flatten coordinates if nested
                if (Array.isArray(coords[0]) && Array.isArray(coords[0][0])) {
                    coords = coords[0];
                }

                var polygonId = _this.addPolygon({
                    id: polygonData.id,
                    coords: coords,
                    color: polygonData.style.color,
                    weight: polygonData.style.weight,
                    fillColor: polygonData.style.fillColor,
                    fillOpacity: polygonData.style.fillOpacity,
                    opacity: polygonData.style.opacity,
                });

                if (polygonId) {
                    importedIds.push(polygonId);
                }
            }
        });

        return importedIds;
    };

    this.removeCircles = function () {
        var _this = this;
        this.mapLet.eachLayer(function (layer) {
            if (layer instanceof L.Circle) {
                _this.mapLet.removeLayer(layer);
            }
        });
    };
};

function rand_id() {
    return Math.random().toString(36).substr(2, 9);
}

function setupLeafletDrawLocalization() {
    // Check if L.drawLocal exists
    if (typeof L.drawLocal === "undefined") {
        console.warn(
            "L.drawLocal is not available. Leaflet.draw may not be loaded."
        );
        return;
    }

    // Complete Turkish translations for Leaflet.draw
    L.drawLocal = {
        draw: {
            toolbar: {
                actions: {
                    title: __("Cancel drawing"),
                    text: __("Cancel"),
                },
                finish: {
                    title: __("Finish drawing"),
                    text: __("Finish"),
                },
                undo: {
                    title: __("Delete last point drawn"),
                    text: __("Delete last point"),
                },
                buttons: {
                    polyline: __("Draw a polyline"),
                    polygon: __("Draw a polygon"),
                    rectangle: __("Draw a rectangle"),
                    circle: __("Draw a circle"),
                    marker: __("Add a marker"),
                    circlemarker: __("Add a circle marker"),
                },
            },
            handlers: {
                circle: {
                    tooltip: {
                        start: __("Click and drag to draw circle"),
                    },
                    radius: __("Radius"),
                },
                circlemarker: {
                    tooltip: {
                        start: __("Click map to place circle marker"),
                    },
                },
                marker: {
                    tooltip: {
                        start: __("Click map to place marker"),
                    },
                },
                polygon: {
                    error: __("Shape edges cannot cross"),
                    tooltip: {
                        start: __("Click to start drawing shape"),
                        cont: __("Click to continue drawing shape"),
                        end: __("Click first point to close this shape"),
                    },
                },
                polyline: {
                    error: __("Shape edges cannot cross"),
                    tooltip: {
                        start: __("Click to start drawing line"),
                        cont: __("Click to continue drawing line"),
                        end: __("Click last point to finish line"),
                    },
                },
                rectangle: {
                    tooltip: {
                        start: __("Click and drag to draw rectangle"),
                    },
                },
                simpleshape: {
                    tooltip: {
                        end: __("Release mouse to finish drawing"),
                    },
                },
            },
        },
        edit: {
            toolbar: {
                actions: {
                    save: {
                        title: __("Save changes"),
                        text: __("Save"),
                    },
                    cancel: {
                        title: __("Cancel editing, discards all changes"),
                        text: __("Cancel"),
                    },
                    clearAll: {
                        title: __("Clear all layers"),
                        text: __("Clear All"),
                    },
                },
                buttons: {
                    edit: __("Edit layers"),
                    editDisabled: __("No layers to edit"),
                    remove: __("Delete layers"),
                    removeDisabled: __("No layers to delete"),
                },
            },
            handlers: {
                edit: {
                    tooltip: {
                        text: __("Drag handles or markers to edit features"),
                        subtext: __("Click cancel to undo changes"),
                    },
                },
                remove: {
                    tooltip: {
                        text: __("Click on a feature to remove"),
                    },
                },
            },
        },
    };
}
