<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\DB;

return new class extends Migration
{
    /**
     * Run the migrations.
     * 
     * This migration imports the database structure from vtc_database_table.sql
     */
    public function up(): void
    {
        // Import SQL file from root directory
        $sqlPath = base_path('vtc_database_table.sql');
        
        if (!file_exists($sqlPath)) {
            throw new \Exception("SQL file not found at: {$sqlPath}");
        }
        
        // Read SQL file
        $sql = file_get_contents($sqlPath);
        
        // Remove SQL comments and split into individual statements
        $sql = preg_replace('/^--.*$/m', '', $sql);
        $sql = preg_replace('/\/\*.*?\*\//s', '', $sql);
        
        // Split by semicolon and execute each statement
        $statements = array_filter(
            array_map('trim', explode(';', $sql)),
            function($stmt) {
                return !empty($stmt) && 
                       stripos($stmt, 'SET') !== 0 && 
                       stripos($stmt, 'ALTER TABLE') !== 0;
            }
        );
        
        foreach ($statements as $statement) {
            if (!empty($statement)) {
                try {
                    DB::statement($statement);
                } catch (\Exception $e) {
                    // Log error but continue (some statements might already exist)
                    \Log::warning("Migration statement skipped: " . substr($statement, 0, 100) . "... Error: " . $e->getMessage());
                }
            }
        }
    }

    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
        // Drop all VTS tables
        Schema::dropIfExists('vts_alarms');
        Schema::dropIfExists('vts_cache');
        Schema::dropIfExists('vts_categories');
        Schema::dropIfExists('vts_category_devices');
        Schema::dropIfExists('vts_devices');
        Schema::dropIfExists('vts_device_logs');
        Schema::dropIfExists('vts_general_settings');
        Schema::dropIfExists('vts_member_types');
        Schema::dropIfExists('vts_permissions');
        Schema::dropIfExists('vts_places');
        Schema::dropIfExists('vts_place_devices');
        Schema::dropIfExists('vts_regions');
        Schema::dropIfExists('vts_region_devices');
        Schema::dropIfExists('vts_sessions');
        Schema::dropIfExists('vts_users');
        Schema::dropIfExists('vts_user_permissions');
    }
};

