<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\GeneralSetting;
use App\Models\User;
use App\Helpers\PublicHelper;
use Illuminate\Support\Facades\Log;

class GeneralSettingController extends Controller
{

    /**
     * Get all settings for the current merchant
     */
    public function index(Request $request)
    {
        //Log::info("getAllSettings called", [$request->all()]);
        try {
            // Ensure default settings exist
            GeneralSetting::ensureDefaultSettingsExist();
            
            $tokenModel = PublicHelper::DecodeJWT($request);            
            $mid = $tokenModel['userId'];
            $storeId = $tokenModel['storeId'];

            $category = $request->input('category', 'all');
            
            if ($category === 'all') {
                $settings = GeneralSetting::getAllSettings($mid, $storeId);
            } else {
                $settings = GeneralSetting::getByCategory($mid, $category, $storeId);
            }

            return response()->json([
                'success' => true,
                'data' => $settings
            ]);

        } catch (\Exception $e) {
            Log::error('Error fetching general settings: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Error fetching settings'
            ], 500);
        }
    }

    /**
     * Get settings by category
     */
    public function getByCategory(Request $request, $category)
    {
        try {
            $tokenModel = PublicHelper::DecodeJWT($request);
            $mid = $tokenModel['userId'];
            $storeId = $tokenModel['storeId'];

            $settings = GeneralSetting::getByCategory($mid, $category, $storeId);

            return response()->json([
                'success' => true,
                'data' => $settings
            ]);

        } catch (\Exception $e) {
            Log::error('Error fetching settings by category: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Error fetching settings'
            ], 500);
        }
    }

    /**
     * Get a specific setting value
     */
    public function getValue(Request $request, $key)
    {
        try {
            $tokenModel = PublicHelper::DecodeJWT($request);
            $mid = $tokenModel['userId'];
            $storeId = $tokenModel['storeId'];

            $default = $request->input('default');
            $value = GeneralSetting::getValue($mid, $key, $default, $storeId);

            return response()->json([
                'success' => true,
                'data' => [
                    'key' => $key,
                    'value' => $value
                ]
            ]);

        } catch (\Exception $e) {
            Log::error('Error fetching setting value: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Error fetching setting value'
            ], 500);
        }
    }

    /**
     * Set a setting value
     */
    public function setValue(Request $request)
    {
        try {
            $tokenModel = PublicHelper::DecodeJWT($request);
            $mid = $tokenModel['userId'];
            $storeId = $tokenModel['storeId'];

            $request->validate([
                'key' => 'required|string|max:100',
                'value' => 'required',
                'type' => 'string|in:string,number,boolean,json',
                'category' => 'string|max:50',
                'description' => 'string|max:255'
            ]);

            $setting = GeneralSetting::setValue(
                $mid,
                $request->input('key'),
                $request->input('value'),
                $request->input('type', 'string'),
                $request->input('category', 'general'),
                $request->input('description'),
                $storeId
            );

            return response()->json([
                'success' => true,
                'data' => $setting,
                'message' => 'Setting updated successfully'
            ]);

        } catch (\Exception $e) {
            Log::error('Error setting value: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Error updating setting'
            ], 500);
        }
    }

    /**
     * Update multiple settings at once
     */
    public function updateMultiple(Request $request)
    {
        //Log::info("updateMultiple called", $request->all());
        try {
            $tokenModel = PublicHelper::DecodeJWT($request);
            $mid = $tokenModel['userId'];
            $storeId = $tokenModel['storeId'];

            $request->validate([
                'settings' => 'required|array',
                'settings.*.key' => 'required|string|max:100',
                'settings.*.value' => 'required',
                'settings.*.type' => 'string|in:string,number,boolean,json',
                'settings.*.category' => 'string|max:50',
                'settings.*.description' => 'string|max:255'
            ]);

            $category = $request->input('category', 'general');
            $updatedSettings = [];
            foreach ($request->input('settings') as $settingData) {
                
                //Log::info("settingData", $settingData);

                $setting = GeneralSetting::setValue(
                    $mid,
                    $settingData['key'],
                    $settingData['value'],
                    $settingData['type'] ?? 'string',
                    $category,
                    $settingData['description'] ?? null,
                    $storeId
                );
                $updatedSettings[] = $setting;
            }

            $response = response()->json([
                'success' => true,
                'data' => $updatedSettings,
                'message' => 'Settings updated successfully'
            ]);

            // If system settings were updated, refresh JWT token
            if ($category === 'system') {
                $this->refreshJwtToken($request, $response);
            }

            return $response;

        } catch (\Exception $e) {
            Log::error('Error updating multiple settings: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Error updating settings'
            ], 500);
        }
    }

    /**
     * Delete a setting
     */
    public function delete(Request $request, $key)
    {
        try {
            $tokenModel = PublicHelper::DecodeJWT($request);
            $mid = $tokenModel['userId'];
            $storeId = $tokenModel['storeId'];

            $deleted = GeneralSetting::deleteSetting($mid, $key, $storeId);

            return response()->json([
                'success' => true,
                'message' => 'Setting deleted successfully',
                'deleted' => $deleted
            ]);

        } catch (\Exception $e) {
            Log::error('Error deleting setting: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Error deleting setting'
            ], 500);
        }
    }

    /**
     * Get fuel prices
     */
    public function getFuelPrices(Request $request)
    {
        try {
            $tokenModel = PublicHelper::DecodeJWT($request);
            $mid = $tokenModel['userId'];
            $storeId = $tokenModel['storeId'];

            $fuelType = $request->input('fuel_type', 'all');
            
            if ($fuelType === 'all') {
                $prices = GeneralSetting::getAllFuelPrices($mid, $storeId);
            } else {
                $price = GeneralSetting::getFuelPrice($mid, $fuelType, $storeId);
                $prices = [$fuelType => $price];
            }

            return response()->json([
                'success' => true,
                'data' => $prices
            ]);

        } catch (\Exception $e) {
            Log::error('Error fetching fuel prices: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Error fetching fuel prices'
            ], 500);
        }
    }

    /**
     * Set fuel price
     */
    public function setFuelPrice(Request $request)
    {
        try {
            $tokenModel = PublicHelper::DecodeJWT($request);
            $mid = $tokenModel['userId'];
            $storeId = $tokenModel['storeId'];

            $request->validate([
                'fuel_type' => 'required|string|in:diesel,gasoline,lpg',
                'price' => 'required|numeric|min:0'
            ]);

            $setting = GeneralSetting::setFuelPrice(
                $mid,
                $request->input('fuel_type'),
                $request->input('price'),
                $storeId
            );

            return response()->json([
                'success' => true,
                'data' => $setting,
                'message' => 'Fuel price updated successfully'
            ]);

        } catch (\Exception $e) {
            Log::error('Error setting fuel price: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Error updating fuel price'
            ], 500);
        }
    }

    /**
     * Get maintenance settings
     */
    public function getMaintenanceSettings(Request $request)
    {
        try {
            $tokenModel = PublicHelper::DecodeJWT($request);
            $mid = $tokenModel['userId'];
            $storeId = $tokenModel['storeId'];

            $settings = GeneralSetting::getMaintenanceSettings($mid, $storeId);

            return response()->json([
                'success' => true,
                'data' => $settings
            ]);

        } catch (\Exception $e) {
            Log::error('Error fetching maintenance settings: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Error fetching maintenance settings'
            ], 500);
        }
    }

    /**
     * Get alert settings
     */
    public function getAlertSettings(Request $request)
    {
        try {
            $tokenModel = PublicHelper::DecodeJWT($request);
            $mid = $tokenModel['userId'];
            $storeId = $tokenModel['storeId'];

            $settings = GeneralSetting::getAlertSettings($mid, $storeId);

            return response()->json([
                'success' => true,
                'data' => $settings
            ]);

        } catch (\Exception $e) {
            Log::error('Error fetching alert settings: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Error fetching alert settings'
            ], 500);
        }
    }

    /**
     * Get system settings
     */
    public function getSystemSettings(Request $request)
    {
        try {
            $tokenModel = PublicHelper::DecodeJWT($request);
            $mid = request()->input('mid');
            $storeId = request()->input('sid');

            $settings = GeneralSetting::getSystemSettings($mid, $storeId);

            return response()->json([
                'success' => true,
                'data' => $settings
            ]);

        } catch (\Exception $e) {
            Log::error('Error fetching system settings: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Error fetching system settings'
            ], 500);
        }
    }

    /**
     * Store multiple settings (alias for updateMultiple)
     */
    public function storeMultiple(Request $request)
    {
        //\Log::info("storeMultiple called", $request->all());
        return $this->updateMultiple($request);
    }
    
    /**
     * Refresh JWT token with updated system settings
     */
    private function refreshJwtToken(Request $request, $response)
    {
        try {
            $tokenModel = PublicHelper::DecodeJWT($request);
            
            if (!$tokenModel) {
                return;
            }

            $user = User::find($tokenModel['userId']);
            
            if (!$user) {
                Log::error('User not found for token refresh');
                return;
            }

            // Create new JWT token using LoginController instance
            $loginController = new \App\Http\Controllers\Auth\LoginController();
            $newToken = $loginController->createJwtToken($user);
                        
            // Add the new token to the existing response
            $response->cookie(config('jwt.public_key'), $newToken, 1440, '/', null, false, true);
            
            // Get updated system settings for locale check
            $systemSettings = \App\Models\GeneralSetting::getSystemSettings(null, $tokenModel['storeId']);
            $newLocale = $systemSettings->get('language')?->typed_value ?? 'tr';
            
            // Update application locale if language was changed
            if ($newLocale !== $tokenModel['locale']) {
                \Illuminate\Support\Facades\App::setLocale($newLocale);
            }
            
        } catch (\Exception $e) {
            Log::error('Error refreshing JWT token: ' . $e->getMessage());
        }
    }
}
