<?php

namespace App\Http\Controllers;
use Illuminate\Support\Facades\DB;  // DB facade'ını burada import ediyoruz
use Illuminate\Support\Facades\Log; // Log facade'ını burada import ediyoruz
use Illuminate\Http\Request;
use App\Traits\ApiResponse;
use App\Http\Resources\DeviceResource;
use App\Http\Resources\DeviceCoordinatesResource;
use App\Http\Resources\DeviceCategoryResource;
use App\Http\Resources\CategoryResource;
use App\Http\Controllers\RegionController;
use App\Http\Controllers\PlaceController;
use App\Http\Resources\DeviceLogResourse;
use App\Models\DeviceLog;
use App\Helpers\Base64Helper;
use App\Helpers\PublicHelper;
use App\Models\Device;

class DeviceController extends Controller
{
    private $deviceStatus = 3; // Aktive devices

    /**
     * get all the data the vehicles and map needs
     */
    public function apiVehiclesMap(Request $request)
    {
        $vehicles = $this->getVehiclesList($request);
        $categories = app(CategoryController::class)->list($request);
        $categoriesDevicesMatch = app(CategoryController::class)->getCategoriesDevicesMatch($request);
        $regions = app(RegionController::class)->list($request);
        $places = app(PlaceController::class)->list($request);

        $data = [
            'vehicles' => $vehicles->getData(true)['data'] ?? [],
            'categories' => $categories->getData(true)['data'] ?? [],
            'categoriesDevicesMatch' => $categoriesDevicesMatch->getData(true)['data'] ?? [],
            'regions' => $regions->getData(true)['data'] ?? [],
            'places' => $places->getData(true)['data'] ?? [],
        ];
        
        return response()->json([
            'success' => true,
            'data' => $data,
        ]);
    }

    /**
     * get vehicle-map blade data
     */
    public function getVehicleMap(Request $request)
    {
        $tokenModel = PublicHelper::DecodeJWT($request);
        $mid = $tokenModel['owenerId'];

        //Log::info('DeviceController>getVehicleMap: ', ['request' => $request->input('data')]);

        $decodedUrl = Base64Helper::decode($request->input('data'));

        //Log::info('DeviceController>getVehicleMap: ', ['decodedUrl' => $decodedUrl]);

        $id = $decodedUrl['did'];
        $date = $decodedUrl['date'];
        $_start_time = $decodedUrl['start_time'];
        $_end_time = $decodedUrl['end_time'];
        $start_time = date('Y-m-d H:i:00', strtotime($date . ' ' . $_start_time));
        $end_time = date('Y-m-d H:i:59', strtotime($date . ' ' . $_end_time));

        //Log::info('DeviceController>getVehicleMap: ', ['start_time' => $start_time, 'end_time' => $end_time]);

        $getVehiclesLogsRequest = $request->merge([
            'did' => $id,
            'start_date' => $start_time,
            'end_date' => $end_time
        ]);
        $getCoordinates = $this->getVehiclesLogs($getVehiclesLogsRequest);
        $coordinates = $getCoordinates->getData(true)['data'] ?? [];


        $vehicleSql = <<<SQL
           SELECT
                d.nm
            FROM vts_devices d
            WHERE
                d.did = :did
            LIMIT 1
        SQL;

        $vehicle111 = DB::select($vehicleSql, [
            'did' => $id
        ]);

        $vehicle = $this->getVehiclesListSimple($request);

        // Regions
        $regions = app(RegionController::class)->list($request);
        // Place
        $places = app(PlaceController::class)->list($request);

        //Log::info("vehicleData", ['start_time' => $start_time, 'endDateTime' => $end_time, 'did' => $id]);

        $data = [
            'coordinates' => $coordinates,
            'vehicle' => $vehicle->getData(true)['data'] ?? [],
            'regions' => $regions->getData(true)['data'] ?? [],
            'places' => $places->getData(true)['data'] ?? [],
        ];

        return response()->json([
            'success' => true,
            'data' => $data,
        ]);

    }

    /**
     * get vehicle/vehicles logs/coordinate
     */
    public function getVehiclesLogs(Request $request){

        $tokenModel = PublicHelper::DecodeJWT($request);
        $mid = $tokenModel['owenerId'];

        //Log::info('DeviceController>getVehiclesLogs: ', ['request' => $request->all()]);

        $did = $request->input('did');
        $startDate = $request->input('start_date');
        $endDate = $request->input('end_date');
        // Build query
        $deviceLogsQuery = DeviceLog::query();
        
        // If vehicle ID is provided, filter by it
        if ($did) {
            $deviceLogsQuery->where('did', $did);
        } else {
            // If no vehicle selected, get all vehicles for this user
            $userVehicles = Device::where('mid', $mid)
                ->where('sts', 3) // Active vehicles
                ->pluck('did')
                ->toArray();
            
            if (!empty($userVehicles)) {
                $deviceLogsQuery->whereIn('did', $userVehicles);
            }
        }
        
        $deviceLogs = $deviceLogsQuery->whereBetween('dt', [$startDate, $endDate])
            ->orderBy('dt')
            ->lazy(10000);  
            // LAZY DATA NEEDS
            // 1 mounth data needs memory_limit=518MB   NOT 512MB
            //              
            //->collect(); // LazyCollection'ı Collection'a çevirir

        //Log::info('DeviceController>getVehiclesLogs: ', ['deviceLogs' => $deviceLogs]);

        return response()->json([
            'success' => true,
            'data' => DeviceLogResourse::collection($deviceLogs, $tokenModel),
        ]);

    }

    public function getVehiclesList(Request $request)
    {
        $tokenModel = PublicHelper::DecodeJWT($request);
        $mid = $tokenModel['owenerId'];
        //Log::info('Vehicles data: ', ['mid' => $mid]);

        // Step 1: Create temporary table with index for better performance
        $createTempTableSql = <<<SQL
            CREATE TEMPORARY TABLE temp_devices_list AS
            SELECT id, did
            FROM vts_devices
            WHERE mid = ? AND sts = ?  -- musteride aktıf olan cihazlar;
        SQL;
        // Geçici tablo oluşturma sorgusunu çalıştır
        $temp = DB::statement($createTempTableSql, [$mid, $this->deviceStatus]);

        // Step 2: Create temporary table for latest logs with optimized approach
        $createLatestLogsSql = <<<SQL
            CREATE TEMPORARY TABLE temp_latest_logs AS
            SELECT 
                dl1.id,
                dl1.log_data_json,
                dl1.latitude,
                dl1.longitude,
                dl1.did,
                dl1.log_datetime
            FROM vts_device_logs dl1
            INNER JOIN (
                SELECT did, MAX(id) as max_id
                FROM vts_device_logs
                WHERE did IN (SELECT did FROM temp_devices_list)
                GROUP BY did
            ) dl2 ON dl1.did = dl2.did AND dl1.id = dl2.max_id
        SQL;
        
        DB::statement($createLatestLogsSql);

        // Geçici tabloyu kullanarak ana sorguyu çalıştırma
        $mainQuerySql = <<<SQL
            SELECT
                d.id,
                l.log_datetime,
                l.log_data_json,
                l.latitude,
                l.longitude,
                d.did,
                d.nm,
                d.extra_data_json,
                d.ord,
                d.sts,
                d.typ,
                d.own,
                d.category_ids_json,
                d.usr,
                d.ga,
                d.sens,
                d.gsm,
                d.imei,
                d.sid,
                d.mid,
                d.is_engine_blocked,
                d.created_at,
                d.updated_at,
                d.created_by,
                d.updated_by
            FROM
                vts_devices d
            LEFT JOIN temp_latest_logs l ON l.did = d.did
            WHERE
                d.did in (SELECT did FROM temp_devices_list)
                AND d.mid = ?
                AND d.sts = ?
            ORDER BY
                d.ord ASC;
        SQL;

        // Ana sorguyu çalıştır ve sonuçları al
        $vehicles = DB::select($mainQuerySql, [$mid, $this->deviceStatus]);

        // Log ile veriyi kontrol edin
        //Log::info('Vehicles data: ', ['vehicles' => $vehicles]);

        $data = [
            'vehicles' => DeviceResource::collection($vehicles, $tokenModel),
        ];

        return response()->json([
            'success' => true,
            'data' => $data,
        ]);

    }

    public function getVehicle(Request $request)
    {
        $tokenModel = PublicHelper::DecodeJWT($request);
        $mid = $tokenModel['owenerId'];

        //Log::info('Vehicles data: ', ['mid' => $mid]);

        $decodedUrl = Base64Helper::decode($request->input('data'));

        //Log::info('DeviceController>getVehicle: ', ['decodedUrl' => $decodedUrl]);

        $did = $decodedUrl['did'];

        // Step 1: Create temporary table with index for better performance
        $createTempTableSql = <<<SQL
            CREATE TEMPORARY TABLE temp_devices_list AS
            SELECT id, did
            FROM vts_devices
            WHERE mid = ? AND did = ? AND sts = ?  -- musteride aktıf olan cihazlar;
        SQL;
        $temp = DB::statement($createTempTableSql, [$mid, $did, $this->deviceStatus]);

        // Step 2: Create temporary table for latest logs with optimized approach
        $createLatestLogsSql = <<<SQL
            CREATE TEMPORARY TABLE temp_latest_logs AS
            SELECT 
                dl1.id,
                dl1.log_data_json,
                dl1.latitude,
                dl1.longitude,
                dl1.did,
                dl1.log_datetime
            FROM vts_device_logs dl1
            INNER JOIN (
                SELECT did, MAX(id) as max_id
                FROM vts_device_logs
                WHERE did IN (SELECT did FROM temp_devices_list)
                GROUP BY did
            ) dl2 ON dl1.did = dl2.did AND dl1.id = dl2.max_id
        SQL;
        DB::statement($createLatestLogsSql);

        // Geçici tabloyu kullanarak ana sorguyu çalıştırma
        $mainQuerySql = <<<SQL
            SELECT
            d.id,
                l.log_datetime,
                l.log_data_json,
                l.latitude,
                l.longitude,
                d.did,
                d.nm,
                d.extra_data_json,
                d.ord,
                d.sts,
                d.typ,
                d.own,
                d.category_ids_json,
                d.usr,
                d.ga,
                d.sens,
                d.gsm,
                d.imei,
                d.sid,
                d.mid,
                d.is_engine_blocked,
                d.created_at,
                d.updated_at,
                d.created_by,
                d.updated_by
            FROM
                vts_devices d
            LEFT JOIN temp_latest_logs l ON l.did = d.did
            WHERE
                d.did IN (SELECT did FROM temp_devices_list)
            LIMIT 1;
        SQL;

        // Ana sorguyu çalıştır ve sonuçları al
        $vehicle = DB::select($mainQuerySql);

        // Log ile veriyi kontrol edin
        //Log::info('vehicle data: ', ['vehicle' => $vehicle]);

        return response()->json([
            'success' => true,
            'data' => DeviceResource::collection($vehicle, $tokenModel),
        ]);

    }

    /**
     * get vehicles list simple
     */
    public function getVehiclesListSimple(Request $request)
    {
        $tokenModel = PublicHelper::DecodeJWT($request);
        $mid = $tokenModel['owenerId'];

        $data = $request->input('data');
        
        $decodedUrl = Base64Helper::decode($data);

        $did = $decodedUrl['did'] ?? null;

        $params['mid'] = $mid;
        $didQuery = "";
        if($did){
            $didQuery = " AND d.did = :did";
            $params['did'] = $did;
        }

        $vehicles = DB::select("
            SELECT 
                d.did as device_did, 
                d.nm as device_name, 
                d.id as device_id
            FROM vts_devices d
              WHERE d.mid = :mid AND d.sts = 3 $didQuery
            ORDER BY d.nm ASC
        ", $params);

        return response()->json([
            'success' => true,
            'data' => $vehicles
        ]);
    }

    /**
     * get vehicles list with alarm
     */
    public function getVehiclesListWithAlarm(Request $request)
    {
        $tokenModel = PublicHelper::DecodeJWT($request);
        $mid = $tokenModel['owenerId'];
        $alarm_id = $request->input('alarm_id');

        $alarm_id_query = "";
        $params = [];
        
        if($alarm_id){
            // Handle comma-separated alarm IDs safely
            $alarmIds = array_map('trim', explode(',', $alarm_id));
            $placeholders = str_repeat('?,', count($alarmIds) - 1) . '?';
            $alarm_id_query = " AND a.aid IN ($placeholders)";
            $params = array_merge($params, $alarmIds);
        }
        
        $params[] = $mid;

        $vehicles = DB::select("
            SELECT 
                d.did, 
                d.nm as name, 
                d.id, 
                d.sts,
                d.created_at,
                d.updated_at,
                a.aid,
                d.extra_data_json as device_settings,
                CASE WHEN a.id IS NOT NULL THEN 1 ELSE 0 END as has_alarm,
                a.settings_json as alarm_settings
            FROM vts_devices d
            LEFT JOIN vts_device_alarms a ON d.did = a.did $alarm_id_query
            WHERE 
                d.mid = ? 
                AND d.sts = 3  
            ORDER BY d.nm ASC
        ", $params);

        return response()->json([
            'success' => true,
            'data' => [
                'vehicles' => $vehicles
            ]
        ]);
    } 

    /**
     * get vehicles list with region
     */
    public function getVehiclesListWithRegion(Request $request)
    {
        $tokenModel = PublicHelper::DecodeJWT($request);
        $mid = $tokenModel['owenerId'];
        $region_id = $request->input('region_id', 0);
        
        $params = [$mid];
        $reginIdQuery = "";
        if($region_id){
            $reginIdQuery = " AND r.pid = ?";
            $params[] = $region_id;
        }

        $vehicles = DB::select("
            SELECT 
                d.did as device_did, 
                d.nm as device_name,
                ( select COUNT(r.id) FROM vts_region_devices r WHERE d.did = r.did ) AS count_region
            FROM vts_devices d 
            WHERE d.mid = ? AND d.sts = 3 $reginIdQuery
            ORDER BY d.nm ASC
        ", $params);

        return response()->json([
            'success' => true,
            'data' => [
                'vehicles' => $vehicles
            ]
        ]);
    }

    /**
     * Update vehicle settings
     */
    public function updateVehicleSettings(Request $request)
    {
        try {
            $tokenModel = PublicHelper::DecodeJWT($request);
            $mid = $tokenModel['owenerId'];

            $data = $request->has('data') ? 
                Base64Helper::decode($request->input('data')) : 
                $request->all();

            // Check if data is an array (multiple vehicles) or single vehicle
            if (isset($data[0])) {
                // Multiple vehicles
                $updatedVehicles = [];
                $errors = [];

                foreach ($data as $vehicleData) {
                    $deviceId = $vehicleData['device_id'] ?? null;
                    $fuelConsumption = $vehicleData['fuel_consumption'] ?? null;
                    $fuelType = $vehicleData['fuel_type'] ?? null;

                    if (!$deviceId || !$fuelConsumption || !$fuelType) {
                        $errors[] = "Missing required data for vehicle ID: $deviceId";
                        continue;
                    }

                    // Get current vehicle data
                    $vehicle = Device::where('did', $deviceId)
                        ->where('mid', $mid)
                        ->where('sts', 3)
                        ->first();

                    if (!$vehicle) {
                        $errors[] = "Vehicle not found for ID: $deviceId";
                        continue;
                    }

                    // Get current extra_data_json data
                    $extraData = $vehicle->extra_data_json ? json_decode($vehicle->extra_data_json, true) : [];

                    // Update extra_data_json data with new values
                    $extraData['lt'] = $fuelConsumption;
                    $extraData['fuel_type'] = $fuelType;

                    // Update vehicle
                    $vehicle->extra_data_json = json_encode($extraData);
                    $vehicle->save();

                    $updatedVehicles[] = $vehicle;
                }

                if (!empty($errors)) {
                    return response()->json([
                        'success' => false,
                        'message' => 'Some vehicles could not be updated: ' . implode(', ', $errors)
                    ], 400);
                }

                return response()->json([
                    'success' => true,
                    'message' => 'Vehicle fuel settings updated successfully',
                    'data' => [
                        'updated_count' => count($updatedVehicles),
                        'vehicles' => $updatedVehicles
                    ]
                ]);

            } else {
                // Single vehicle (existing logic)
                $deviceId = $data['device_id'] ?? null;
                $deviceName = $data['device_name'] ?? null;
                $plate = $data['plate'] ?? null;
                $driver = $data['driver'] ?? null;
                $fuelConsumption = $data['fuel_consumption'] ?? null;
                $fuelType = $data['fuel_type'] ?? null;

                if (!$deviceId) {
                    return response()->json([
                        'success' => false,
                        'message' => 'Device ID is required'
                    ], 400);
                }

                // Get current vehicle data
                $vehicle = Device::where('did', $deviceId)
                    ->where('mid', $mid)
                    ->where('sts', 3)
                    ->first();

                if (!$vehicle) {
                    return response()->json([
                        'success' => false,
                        'message' => 'Vehicle not found'
                    ], 404);
                }

                // Get current extra_data_json data
                $extraData = $vehicle->extra_data_json ? json_decode($vehicle->extra_data_json, true) : [];

                // Update extra_data_json data with new values
                if ($deviceName !== null) {
                    $vehicle->nm = $deviceName;
                    $extraData['nm'] = $deviceName; // Also update in extra_data_json
                }
                if ($plate !== null) {
                    $extraData['plt'] = $plate;
                }
                if ($driver !== null) {
                    $extraData['drvr'] = $driver;
                }
                if ($fuelConsumption !== null) {
                    $extraData['lt'] = $fuelConsumption;
                }
                if ($fuelType !== null) {
                    $extraData['fuel_type'] = $fuelType;
                }

                // Update vehicle
                $vehicle->extra_data_json = json_encode($extraData);
                $vehicle->save();

                return response()->json([
                    'success' => true,
                    'message' => 'Vehicle settings updated successfully',
                    'data' => [
                        'vehicle' => $vehicle
                    ]
                ]);
            }

        } catch (\Exception $e) {
            \Log::error('Error updating vehicle settings: ' . $e->getMessage(), [
                'file' => $e->getFile(),
                'line' => $e->getLine(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Error updating vehicle settings: ' . $e->getMessage()
            ], 500);
        }
    }



}
