<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;
use App\Http\Resources\CategoryResource;
use App\Traits\ApiResponse;
use App\Helpers\Base64Helper;
use App\Helpers\PublicHelper;

class CategoryController extends Controller
{
    use ApiResponse;

    /**
     * Get user ID and store ID from JWT token in the request cookies
     */
    private function getUserIdFromToken(Request $request)
    {
        $tokenModel = PublicHelper::DecodeJWT($request);
        return [
            'userId' => $tokenModel['userId'] ?? null,
            'storeId' => $tokenModel['storeId'] ?? 0,
        ];
    }

    /**
     * List all categories for the current user
     */
    public function list(Request $request)
    {
        $tokenData = $this->getUserIdFromToken($request);
        $mid = $tokenData['userId'];
        $storeId = $tokenData['storeId'];
        
        if (!$mid) {
            return $this->error('Unauthorized', 401);
        }

        $did = $request->input('device_id') ?? null;
        $params = [];
        $matchQuery = '';
        $matchSelect = '';
        $deviceDatabaseId = null;
        
        if ($did) {
            // Convert did (string) to device id (int)
            $device = DB::table('vts_devices')->where('did', $did)->where('mid', $mid)->first();
            if ($device) {
                $deviceDatabaseId = $device->id;
                $params[] = $deviceDatabaseId;
                $matchQuery = " LEFT JOIN vts_category_devices m on m.device_id = ? and m.category_id = c.id ";
                $matchSelect = " CASE WHEN m.device_id IS NOT NULL THEN 1 ELSE 0 END as device_categoty_match, ";
            } else {
                $matchSelect = " 0 as device_categoty_match, ";
            }
        } else {
            $matchSelect = " 0 as device_categoty_match, ";
        }

        $params[] = $mid;

        $categories = DB::select("
            SELECT 
                c.*,
                $matchSelect
                COUNT(cd.device_id) as device_count
            FROM vts_categories c
            LEFT JOIN vts_category_devices cd ON c.id = cd.category_id
            $matchQuery
            WHERE c.mid = ?
            and c.sts = 1
            GROUP BY c.id
            ORDER BY c.ord ASC
        ", $params);

        return response()->json([
            'success' => true,
            'data' => CategoryResource::collection($categories),
        ]);
    }

    function getCategoriesDevicesMatch(Request $request){
        $tokenData = $this->getUserIdFromToken($request);
        $mid = $tokenData['userId'];
        $storeId = $tokenData['storeId'];

        // Step 1: Create temporary table with index for better performance
        $params['mid'] = $mid;
        $params['sts'] = 3;

        $categories = DB::select("
            SELECT 
                c.*
            FROM vts_category_devices c
            inner join vts_devices d on d.id = c.device_id and d.mid = :mid and d.sts = :sts
        ", $params);

        //Log::info("categoriesDevicesMatch", $categories);

        return response()->json([
            'success' => true,
            'data' => $categories,
        ]);
    }

    /**
     * Add a new category
     */
    public function add(Request $request)
    {
        $tokenData = $this->getUserIdFromToken($request);
        $mid = $tokenData['userId'];
        $storeId = $tokenData['storeId'];
        
        if (!$mid) {
            return $this->error('Unauthorized', 401);
        }

        $data = $request->has('data') ?
            Base64Helper::decode($request->input('data')) :
            $request->all();

        // Validate input data
        $validator = \Illuminate\Support\Facades\Validator::make($data, [
            'name' => 'required|string|max:255',
            'order' => 'nullable|integer',
            'status_id' => 'nullable|integer',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => $validator->errors()->first(),
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            // Auto-increment order if not provided
            // Check only active categories with same mid and store_id
            if (is_null($data['order'] ?? null)) {
                $maxOrd = DB::table('vts_categories')
                    ->where('mid', $mid)
                    ->where('sid', $storeId)
                    ->where('sts', 1)
                    ->max('ord');
                $ord = is_null($maxOrd) ? 1 : $maxOrd + 1;
            } else {
                $ord = $data['order'];
            }

            $categoryId = DB::table('vts_categories')->insertGetId([
                'sid' => $storeId,
                'mid' => $mid,
                'lid' => 0, // Location/Language ID
                'sef' => \Illuminate\Support\Str::slug($data['name']), // SEF URL
                'name' => $data['name'],
                'ord' => $ord,
                'pcnt' => 0, // Product count - başlangıçta 0
                'sts' => $data['status_id'] ?? 1,
                'tx' => '', // Extra data
                'created_at' => now(),
                'created_by' => $mid
            ]);

            $category = DB::table('vts_categories')->where('id', $categoryId)->first();

            return response()->json([
                'success' => true,
                'data' => new CategoryResource($category),
                'message' => 'Category added successfully.'
            ]);

        } catch (\Exception $e) {
            Log::error('Error creating category: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Error creating category.'
            ], 500);
        }
    }

    /**
     * Edit an existing category
     */
    public function edit(Request $request)
    {
        $data = $request->has('data') ?
            Base64Helper::decode($request->input('data')) :
            $request->all();

        $id = $data['id'] ?? null;
        $tokenData = $this->getUserIdFromToken($request);
        $mid = $tokenData['userId'];
        $storeId = $tokenData['storeId'];

        $category = DB::table('vts_categories')->where('id', $id)->first();
        if (!$category) {
            return $this->error('Category not found', 404);
        }

        // Check ownership
        if ($category->mid != $mid) {
            return $this->error('Unauthorized', 403);
        }

        // Update fields
        DB::table('vts_categories')->where('id', $id)->update([
            'name' => $data['name'] ?? $category->name,
            'ord' => $data['order'] ?? $category->ord,
            'sts' => $data['status_id'] ?? $category->sts,
            'updated_at' => now(),
            'updated_by' => $mid
        ]);

        $updatedCategory = DB::table('vts_categories')->where('id', $id)->first();

        return response()->json([
            'success' => true,
            'data' => new CategoryResource($updatedCategory),
            'message' => 'Category updated successfully.'
        ]);
    }

    /**
     * Delete a category
     */
    public function delete(Request $request)
    {
        $data = $request->has('data') ?
            Base64Helper::decode($request->input('data')) :
            $request->all();

        $id = $data['id'] ?? null;
        $tokenData = $this->getUserIdFromToken($request);
        $mid = $tokenData['userId'];
        $storeId = $tokenData['storeId'];

        $category = DB::table('vts_categories')->where('id', $id)->first();
        if (!$category) {
            return $this->error('Category not found', 404);
        }

        // Check ownership
        if ($category->mid != $mid) {
            return $this->error('Unauthorized', 403);
        }

        // Delete category device relationships first
        DB::table('vts_category_devices')->where('category_id', $id)->delete();

        // Delete the category
        DB::table('vts_categories')->where('id', $id)->delete();

        // Reorder remaining categories
        $this->reorderCategoriesAfterDelete($mid);

        return response()->json([
            'success' => true,
            'message' => 'Category deleted successfully.'
        ]);
    }

    /**
     * Update categories order
     */
    public function updateOrder(Request $request)
    {
        $tokenData = $this->getUserIdFromToken($request);
        $mid = $tokenData['userId'];
        $storeId = $tokenData['storeId'];
        
        if (!$mid) {
            return $this->error('Unauthorized', 401);
        }

        $data = $request->has('data') ?
            Base64Helper::decode($request->input('data')) :
            $request->all();

        $orders = $data['orders'] ?? $data['order'] ?? [];

        foreach ($orders as $orderData) {
            $category = DB::table('vts_categories')->where('id', $orderData['id'])->first();

            // Check ownership
            if ($category && $category->mid == $mid) {
                DB::table('vts_categories')->where('id', $orderData['id'])->update([
                    'ord' => $orderData['order'] ?? $orderData['ord'],
                    'updated_at' => now(),
                    'updated_by' => $mid
                ]);
            }
        }

        return response()->json([
            'success' => true,
            'message' => 'Category order updated successfully.'
        ]);
    }

    /**
     * Duplicate a category
     */
    public function duplicate(Request $request)
    {
        $tokenData = $this->getUserIdFromToken($request);
        $mid = $tokenData['userId'];
        $storeId = $tokenData['storeId'];
        
        if (!$mid) {
            return $this->error('Unauthorized', 401);
        }

        $data = $request->has('data') ?
            Base64Helper::decode($request->input('data')) :
            $request->all();

        // Validation
        $validator = \Illuminate\Support\Facades\Validator::make($data, [
            'id' => 'required|integer|exists:vts_categories,id',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => $validator->errors()->first(),
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            $originalCategory = DB::table('vts_categories')->where('id', $data['id'])->first();

            if (!$originalCategory) {
                return response()->json([
                    'success' => false,
                    'message' => 'Category not found'
                ], 404);
            }

            // Check ownership
            if ($originalCategory->mid != $mid) {
                return response()->json([
                    'success' => false,
                    'message' => 'Unauthorized'
                ], 403);
            }

            // Get next order number - check active categories with same mid and store_id
            $maxOrd = DB::table('vts_categories')
                ->where('mid', $mid)
                ->where('sid', $storeId)
                ->where('sts', 1)
                ->max('ord');
            $newOrd = is_null($maxOrd) ? 1 : $maxOrd + 1;

            // Create duplicated category with " (Copy)" suffix
            $duplicatedCategoryId = DB::table('vts_categories')->insertGetId([
                'sid' => $storeId,
                'mid' => $mid,
                'lid' => $originalCategory->lid ?? 0,
                'sef' => \Illuminate\Support\Str::slug($originalCategory->name . ' Copy'),
                'name' => $originalCategory->name . ' (Copy)',
                'ord' => $newOrd,
                'pcnt' => 0, // Yeni kategori için device count 0
                'sts' => $originalCategory->sts,
                'tx' => $originalCategory->tx ?? '',
                'created_at' => now(),
                'created_by' => $mid
            ]);

            $duplicatedCategory = DB::table('vts_categories')->where('id', $duplicatedCategoryId)->first();

            return response()->json([
                'success' => true,
                'data' => new CategoryResource($duplicatedCategory),
                'message' => 'Category duplicated successfully.'
            ]);

        } catch (\Exception $e) {
            Log::error('Error duplicating category: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Error duplicating category.'
            ], 500);
        }
    }

    /**
     * Toggle category status (active/inactive)
     */
    public function toggleStatus(Request $request)
    {
        $tokenData = $this->getUserIdFromToken($request);
        $mid = $tokenData['userId'];
        $storeId = $tokenData['storeId'];
        
        if (!$mid) {
            return $this->error('Unauthorized', 401);
        }

        $data = $request->has('data') ?
            Base64Helper::decode($request->input('data')) :
            $request->all();

        $id = $data['id'];
        $status = $data['status_id'] ?? $data['sts'];

        $category = DB::table('vts_categories')->where('id', $id)->first();
        if (!$category) {
            return $this->error('Category not found', 404);
        }

        // Check ownership
        if ($category->mid != $mid) {
            return $this->error('Unauthorized', 403);
        }

        DB::table('vts_categories')->where('id', $id)->update([
            'sts' => $status,
            'updated_at' => now(),
            'updated_by' => $mid
        ]);

        $updatedCategory = DB::table('vts_categories')->where('id', $id)->first();
        $statusText = $status == 1 ? 'activated' : 'deactivated';

        return response()->json([
            'success' => true,
            'data' => new CategoryResource($updatedCategory),
            'message' => "Category {$statusText} successfully."
        ]);
    }

    /**
     * Get category statistics
     */
    public function getStats(Request $request)
    {
        $tokenData = $this->getUserIdFromToken($request);
        $mid = $tokenData['userId'];
        $storeId = $tokenData['storeId'];
        
        if (!$mid) {
            return $this->error('Unauthorized', 401);
        }

        $stats = DB::select("
            SELECT 
                COUNT(*) as total_categories,
                SUM(CASE WHEN sts = 1 THEN 1 ELSE 0 END) as active_categories,
                SUM(CASE WHEN sts = 0 THEN 1 ELSE 0 END) as inactive_categories
            FROM vts_categories 
            WHERE mid = ?
        ", [$mid]);

        return response()->json([
            'success' => true,
            'data' => $stats[0] ?? [
                'total_categories' => 0,
                'active_categories' => 0,
                'inactive_categories' => 0
            ]
        ]);
    }

    /**
     * Reorder categories after deletion
     */
    private function reorderCategoriesAfterDelete($mid)
    {
        $categories = DB::table('vts_categories')
            ->where('mid', $mid)
            ->orderBy('ord', 'asc')
            ->get();

        foreach ($categories as $index => $category) {
            DB::table('vts_categories')
                ->where('id', $category->id)
                ->update(['ord' => $index + 1]);
        }
    }

    /**
     * Set device assignment for category
     */
    public function setDevice(Request $request)
    {
        $tokenData = $this->getUserIdFromToken($request);
        $mid = $tokenData['userId'];
        $storeId = $tokenData['storeId'];
        
        if (!$mid) {
            return $this->error('Unauthorized', 401);
        }

        $data = $request->has('data') ?
            Base64Helper::decode($request->input('data')) :
            $request->all();

        //Log::info($data);

        $categoryId = $data['category_id'] ?? null;
        $deviceId = $data['device_id'] ?? null;
        $status = $data['status'] ?? null;

        if (!$categoryId || !$deviceId || $status === null) {
            return response()->json([
                'success' => false,
                'message' => 'Category ID, Device ID, and status are required.'
            ], 422);
        }

        // Verify category exists and belongs to user
        $category = DB::table('vts_categories')->where('id', $categoryId)->first();
        if (!$category) {
            return response()->json([
                'success' => false,
                'message' => 'Category not found.'
            ], 404);
        }

        if ($category->mid != $mid) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized access to category.'
            ], 403);
        }

        // Verify device exists and belongs to user
        $device = DB::table('vts_devices')->where('did', $deviceId)->first();
        if (!$device) {
            return response()->json([
                'success' => false,
                'message' => 'Device not found.'
            ], 404);
        }

        if ($device->mid != $mid) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized access to device.'
            ], 403);
        }

        // Use device->id (int) instead of did (string)
        $deviceDatabaseId = $device->id;

        try {
            if ($status == 1) {
                // Assign device to category
                DB::table('vts_category_devices')->insertOrIgnore([
                    'category_id' => $categoryId,
                    'device_id' => $deviceDatabaseId,  // Use id (int) not did (string)
                    'created_at' => now(),
                    'created_by' => $mid
                ]);
            } else {
                // Remove device from category
                DB::table('vts_category_devices')
                    ->where('category_id', $categoryId)
                    ->where('device_id', $deviceDatabaseId)  // Use id (int) not did (string)
                    ->delete();
            }

            return response()->json([
                'success' => true,
                'message' => $status == 1 ? 'Device assigned to category successfully.' : 'Device removed from category successfully.'
            ]);

        } catch (\Exception $e) {
            Log::error('Error setting device category assignment: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Error updating device category assignment.'
            ], 500);
        }
    }
} 