<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use App\Models\Alarm;
use App\Http\Resources\AlarmResource;
use App\Helpers\Base64Helper;
use App\Helpers\PublicHelper;
use App\Models\User;

class AlarmController extends Controller
{
    /**
     * Get user ID from JWT token in the request cookies
     */
    private function getUserIdFromToken(Request $request)
    {
        $tokenModel = PublicHelper::DecodeJWT($request);
        return $tokenModel['userId'] ?? null;
    }

    /**
     * List alarms by member and device
     */
    public function list(Request $request)
    {
       $mid = $this->getUserIdFromToken($request);
        if (!$mid) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized'
            ], 401);
        }

        $data = $request->has('data') ?
            Base64Helper::decode($request->input('data')) :
            $request->all();
        
        $alarms = [];

        if (is_array($data) && isset($data['did'])) {
            $did = $data['did'];

            $alarms = Alarm::where('mid', $mid)
                ->where('did', $did)
                ->get();
        }
        return response()->json([
                'success' => true,
                'data' => AlarmResource::collection($alarms),
            ]);
    }

    /**
     * Set or update alarm for a device
     */
    public function setDevice(Request $request)
    {
        $mid = $this->getUserIdFromToken($request);
        if (!$mid) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized'
            ], 401);
        }

        $data = $request->has('data') ?
            Base64Helper::decode($request->input('data')) :
            $request->all();
        
        //Log::info('Alarm data: ', ['data' => $data]);

        $did = $data['device_id'] ?? null;
        $aid = $data['alarm_form_id'] ?? null;
        $settings = $data;
        $status = $data["status_id_$aid"] ?? 0;
        $send_status = $data["send_status_id_$aid"] ?? 0;

        // engine block password check
        if($aid == 302) {
            $password = $data['password'] ?? null;
            if($password !== '') {
                // get member password
                $member = User::find($mid);
                if($member->psw !== $password) {
                    return response()->json([
                        'success' => false,
                        'message' => 'Invalid password'
                    ]);
                }
            }
        }

        $alarm = Alarm::updateOrCreate(
            [
                'mid' => $mid,
                'did' => $did,
                'aid' => $aid,
            ],
            [
                'settings_json' => $settings,
                'sts' => $status,
                'sid' => $send_status,
            ]
        );
        // if alarm was recently created, set created_at and created_by
        if($alarm->wasRecentlyCreated) {
            $alarm->created_at = now();
            $alarm->created_by = $mid;
        }
        else {
            $alarm->updated_at = now();
            $alarm->updated_by = $mid;
        }
        $alarm->save();

        return response()->json([
            'success' => true,
            'data' => new AlarmResource($alarm),
            'message' => 'Alarm set successfully.',
        ]);
    }

    /**
     * Get a single alarm by id
     */
    public function getAlarm(Request $request)
    {
        $id = $request->input('id');
        $alarm = Alarm::find($id);

        if (!$alarm) {
            return response()->json([
                'success' => false,
                'message' => 'Alarm not found.',
            ], 404);
        }

        return response()->json([
            'success' => true,
            'data' => new AlarmResource($alarm),
        ]);
    }
}