<?php $__env->startSection('title', __('Vehicle Health Report')); ?>

<?php $__env->startSection('content'); ?>
<div class="page-wrapper" id="vehicle-health-report">
    <div class="page-header d-print-none">
        <div class="container-xl">
            <div class="row g-2 align-items-center">
                <div class="col">
                    <div class="page-pretitle">
                        <a href="/reports" class="text-muted"><?php echo e(__('Reports')); ?></a> /
                        <?php echo e(__('Vehicle Management Reports')); ?>

                    </div>
                    <h2 class="page-title">
                        <i class="ti ti-heart-pulse me-2"></i>
                        <?php echo e(__('Vehicle Health Report')); ?>

                    </h2>
                    <div class="text-muted mt-1">
                        <?php echo e(__('Comprehensive vehicle health analysis including maintenance, performance, and predictive insights')); ?>

                    </div>
                </div>
                <div class="col-auto ms-auto d-print-none">
                    <div class="btn-list">
                        <button type="button" class="btn btn-primary" onclick="window.print()">
                            <i class="ti ti-printer me-1"></i>
                            <?php echo e(__('Print Report')); ?>

                        </button>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <div class="page-body">
        <div class="container-xl">
            <!-- Filters -->
            <div class="card mb-3">
                <div class="card-header">
                    <h3 class="card-title"><?php echo e(__('Filters')); ?></h3>
                    <div class="card-actions">
                        <button type="button" class="btn btn-sm btn-outline-primary" data-bs-toggle="collapse"
                            data-bs-target="#helpSection">
                            <i class="ti ti-help me-1"></i><?php echo e(__('How to use this report')); ?>

                        </button>
                    </div>
                </div>
                <div class="card-body">
                    <!-- Help Section -->
                    <div class="collapse mb-3" id="helpSection">
                        <div class="alert alert-info">
                            <h4 class="alert-heading"><i class="ti ti-info-circle me-2"></i>
                                <?php echo e(__('How to use the Vehicle Health Report')); ?></h4>
                            <p class="mb-2"><strong><?php echo e(__('Step 1:')); ?></strong>
                                <?php echo e(__('Select a vehicle (optional) or leave as "All Vehicles"')); ?></p>
                            <p class="mb-2"><strong><?php echo e(__('Step 2:')); ?></strong>
                                <?php echo e(__('Choose a health status filter (optional) to see only vehicles in that condition')); ?></p>
                            <p class="mb-2"><strong><?php echo e(__('Step 3:')); ?></strong>
                                <?php echo e(__('Set your date range (when the data was collected)')); ?></p>
                            <p class="mb-2"><strong><?php echo e(__('Step 4:')); ?></strong>
                                <?php echo e(__('Click "Generate Report" to see the results')); ?></p>
                        </div>
                    </div>
                    <div class="row g-3">
                        <div class="col-md-3">
                            <label class="form-label"><?php echo e(__('Vehicle')); ?></label>
                            <div class="input-group">
                                <select class="form-select" id="vehicleSelect">
                                    <option value=""><?php echo e(__('All Vehicles')); ?></option>
                                </select>
                                <button type="button" class="btn btn-outline-secondary d-none" id="vehicleSettingsLink"
                                    onclick="goToVehicleSettings()" title="<?php echo e(__('Go to Vehicle Settings')); ?>">
                                    <i class="icon ti ti-settings"></i>
                                </button>
                            </div>
                        </div>
                        <div class="col-md-2">
                            <label class="form-label"><?php echo e(__('Health Status')); ?></label>
                            <select class="form-select" id="healthStatusFilter">
                                <option value=""><?php echo e(__('All Status')); ?></option>
                                <option value="excellent"><?php echo e(__('Excellent')); ?></option>
                                <option value="good"><?php echo e(__('Good')); ?></option>
                                <option value="fair"><?php echo e(__('Fair')); ?></option>
                                <option value="poor"><?php echo e(__('Poor')); ?></option>
                                <option value="critical"><?php echo e(__('Critical')); ?></option>
                            </select>
                        </div>
                        <div class="col-md-3">
                            <label class="form-label"><?php echo e(__('Date Range')); ?></label>
                            <div class="input-group">
                                <input type="date" class="form-control" id="startDate"
                                    value="<?php echo e(date('Y-m-d', strtotime('-1 month'))); ?>">
                                <span class="input-group-text"><?php echo e(__('to')); ?></span>
                                <input type="date" class="form-control" id="endDate" value="<?php echo e(date('Y-m-d')); ?>">
                            </div>
                        </div>
                        <div class="col-md-2 d-flex align-items-end">
                            <button type="button" class="btn btn-primary w-100" onclick="generateReport()">
                                <i class="icon ti ti-search me-1"></i>
                                <?php echo e(__('Generate Report')); ?>

                            </button>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Health Overview Cards -->
            <div class="row row-cards mb-3 d-none" id="healthOverviewCards">
                <div class="col-sm-6 col-lg-3">
                    <div class="card">
                        <div class="card-body">
                            <div class="d-flex align-items-center">
                                <div class="subheader">
                                    <?php echo e(__('Overall Health Score')); ?>

                                    <i class="ti ti-info-circle text-muted ms-1" data-bs-toggle="tooltip"
                                        title="<?php echo e(__('Average health score of all vehicles. Higher is better.')); ?>"></i>
                                </div>
                                <div class="ms-auto lh-1">
                                    <i class="ti ti-heart-pulse ms-1 text-success" style="font-size: 1.5rem;"></i>
                                </div>
                            </div>
                            <div class="d-flex align-items-baseline">
                                <div class="h1 mb-0 me-2" id="overallHealthScore">-</div>
                                <div class="text-muted" id="healthScoreTrend">-</div>
                            </div>
                            <div id="health-score-chart" style="width: 100%; height: 40px;"></div>
                            <div class="text-muted small" id="healthScoreBreakdown">-</div>
                        </div>
                    </div>
                </div>
                <div class="col-sm-6 col-lg-3">
                    <div class="card">
                        <div class="card-body">
                            <div class="d-flex align-items-center">
                                <div class="subheader">
                                    <?php echo e(__('Maintenance Due')); ?>

                                    <i class="ti ti-info-circle text-muted ms-1" data-bs-toggle="tooltip"
                                        title="<?php echo e(__('Vehicles that need service or maintenance soon.')); ?>"></i>
                                </div>
                                <div class="ms-auto lh-1">
                                    <i class="ti ti-tools ms-1 text-warning" style="font-size: 1.5rem;"></i>
                                </div>
                            </div>
                            <div class="d-flex align-items-baseline">
                                <div class="h1 mb-0 me-2" id="maintenanceDue">-</div>
                            </div>
                            <div id="maintenance-chart" style="width: 100%; height: 40px;"></div>
                            <div class="text-muted small" id="maintenanceBreakdown">-</div>
                        </div>
                    </div>
                </div>
                <div class="col-sm-6 col-lg-3">
                    <div class="card">
                        <div class="card-body">
                            <div class="d-flex align-items-center">
                                <div class="subheader">
                                    <?php echo e(__('Driving Score')); ?>

                                    <i class="ti ti-info-circle text-muted ms-1" data-bs-toggle="tooltip"
                                        title="<?php echo e(__('How well vehicles are being driven based on driving data.')); ?>"></i>
                                </div>
                                <div class="ms-auto lh-1">
                                    <i class="ti ti-trending-up ms-1 text-info" style="font-size: 1.5rem;"></i>
                                </div>
                            </div>
                            <div class="d-flex align-items-baseline">
                                <div class="h1 mb-0 me-2" id="performanceScore">-</div>
                                <div class="text-muted" id="performanceTrend">-</div>
                            </div>
                            <div id="performance-chart" style="width: 100%; height: 40px;"></div>
                            <div class="text-muted small" id="performanceBreakdown">-</div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Health Status Distribution -->
            <div class="row mb-3 d-none" id="healthStatusDistributionRow">
                <div class="col-lg-6">
                    <div class="card">
                        <div class="card-header">
                            <h3 class="card-title">
                                <?php echo e(__('Health Status Distribution')); ?>

                                <i class="ti ti-info-circle text-muted ms-1" data-bs-toggle="tooltip"
                                    title="<?php echo e(__('Shows how many vehicles are in each health category.')); ?>"></i>
                            </h3>
                            <div class="card-actions">
                                <div class="text-muted small">
                                    <?php echo e(__('Excellent: 90-100% | Good: 80-89% | Fair: 70-79% | Poor: 60-69% | Critical:
                                    Below 60%')); ?>

                                </div>
                            </div>
                        </div>
                        <div class="card-body">
                            <div id="healthStatusChart" style="height: 300px;"></div>
                        </div>
                    </div>
                </div>
                <div class="col-lg-6">
                    <div class="card">
                        <div class="card-header">
                            <h3 class="card-title">
                                <?php echo e(__('Health Score Breakdown')); ?>

                                <i class="ti ti-info-circle text-muted ms-1" data-bs-toggle="tooltip"
                                    title="<?php echo e(__('Shows remaining KM, days to inspection, and performance score for each vehicle.')); ?>"></i>
                            </h3>
                            <div class="card-actions">
                                <div class="text-muted small">
                                    <?php echo e(__('Hover over bars to see specific values.')); ?>

                                </div>
                            </div>
                        </div>
                        <div class="card-body">
                            <div id="healthScoreBreakdownChart" style="height: 350px;"></div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Vehicle Health Table -->
            <div class="card mb-3 d-none" id="vehicleHealthTableCard">
                <div class="card-header">
                    <h3 class="card-title">
                        <?php echo e(__('Vehicle Health Details')); ?>

                        <i class="ti ti-info-circle text-muted ms-1" data-bs-toggle="tooltip"
                            title="<?php echo e(__('Detailed health information for each vehicle. Click Details to see more.')); ?>"></i>
                    </h3>
                    <div class="card-actions">
                        <div class="text-muted small">
                            <?php echo e(__('Click the Details button to see detailed information about each vehicle.')); ?>

                        </div>
                    </div>
                </div>
                <div class="card-body p-0">
                    <div id="vehicle-health-table"></div>
                </div>
            </div>

            <!-- Performance Analysis -->
            <div class="row mb-3 d-none" id="performanceAnalysisRow">
                <div class="col-lg-6">
                    <div class="card">
                        <div class="card-header">
                            <h3 class="card-title">
                                <?php echo e(__('Engine Performance Trend')); ?>

                                <i class="ti ti-info-circle text-muted ms-1" data-bs-toggle="tooltip"
                                    title="<?php echo e(__('How efficiently vehicles are running over time. Higher = better performance.')); ?>"></i>
                            </h3>
                            <div class="card-actions">
                                <div class="text-muted small">
                                    <?php echo e(__('Shows engine efficiency based on speed and driving patterns.')); ?>

                                </div>
                            </div>
                        </div>
                        <div class="card-body">
                            <div id="enginePerformanceChart" style="height: 300px;"></div>
                        </div>
                    </div>
                </div>
                <div class="col-lg-6">
                    <div class="card">
                        <div class="card-header">
                            <h3 class="card-title">
                                <?php echo e(__('Driving Behavior Impact')); ?>

                                <i class="ti ti-info-circle text-muted ms-1" data-bs-toggle="tooltip"
                                    title="<?php echo e(__('Number of harsh driving events (sudden braking, acceleration). Lower = better driving.')); ?>"></i>
                            </h3>
                            <div class="card-actions">
                                <div class="text-muted small">
                                    <?php echo e(__('Harsh driving events can damage vehicles and increase fuel costs.')); ?>

                                </div>
                            </div>
                        </div>
                        <div class="card-body">
                            <div id="drivingBehaviorChart" style="height: 300px;"></div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Loading State -->
            <div class="text-center py-4 d-none" id="loadingState">
                <div class="spinner-border text-primary" role="status">
                    <span class="visually-hidden"><?php echo e(__('Loading...')); ?></span>
                </div>
                <div class="mt-2"><?php echo e(__('Generating vehicle health report...')); ?></div>
            </div>

            <!-- No Data State -->
            <div class="text-center py-4 d-none" id="noDataState">
                <div class="empty">
                    <div class="empty-img">
                        <i class="ti ti-heart-pulse text-muted" style="font-size: 3rem;"></i>
                    </div>
                    <p class="empty-title"><?php echo e(__('No vehicle health data found')); ?></p>
                    <p class="empty-subtitle text-muted">
                        <?php echo e(__('No vehicle health data was found for the selected period.')); ?>

                    </p>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Vehicle Details Modal -->
<div class="modal fade" id="vehicleDetailsModal" tabindex="-1" aria-labelledby="vehicleDetailsModalLabel"
    aria-hidden="true">
    <div class="modal-dialog modal-xl modal-dialog-centered">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="vehicleDetailsModalLabel"><?php echo e(__('Vehicle Health Details')); ?></h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
                <div id="vehicleDetailsContent">
                    <!-- Vehicle details will be populated here -->
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal"><?php echo e(__('Close')); ?></button>
            </div>
        </div>
    </div>
</div>
<?php $__env->stopSection(); ?>

<?php $__env->startSection('styles'); ?>
<style>
    .health-status-excellent {
        color: #28a745;
    }

    .health-status-good {
        color: #17a2b8;
    }

    .health-status-fair {
        color: #ffc107;
    }

    .health-status-poor {
        color: #fd7e14;
    }

    .health-status-critical {
        color: #dc3545;
    }

    .health-score-badge {
        font-size: 0.875rem;
        padding: 0.25rem 0.5rem;
        border-radius: 0.25rem;
        font-weight: 600;
    }

    .health-score-excellent {
        background-color: #d4edda;
        color: #155724;
    }

    .health-score-good {
        background-color: #d1ecf1;
        color: #0c5460;
    }

    .health-score-fair {
        background-color: #fff3cd;
        color: #856404;
    }

    .health-score-poor {
        background-color: #ffeaa7;
        color: #d63031;
    }

    .health-score-critical {
        background-color: #f8d7da;
        color: #721c24;
    }
</style>
<?php $__env->stopSection(); ?>

<?php $__env->startSection('scripts'); ?>
<script>
    var speedUnit = "";
    var lengthUnit = "";
    var currency = "";
    
    let reportData = [];
    let vehicleHealthTable;
    let vehicleList = [];
    
    // Initialize page state
    $('#healthOverviewCards').addClass('d-none');
    $('#vehicleHealthTableCard').addClass('d-none');
    $('#loadingState').addClass('d-none');
    $('#noDataState').addClass('d-none');
    $('#healthStatusDistributionRow').addClass('d-none');
    $('#performanceAnalysisRow').addClass('d-none');
    
    // Wait for app to be fully initialized (translations + session settings)
    onAppReady(function() {        
        // Initialize the page after app is ready
        initializePage();
    });
    
    function initializePage() {
        // Load session settings
        speedUnit = sessionSettings.speedUnit || 'km/h';
        lengthUnit = sessionSettings.lengthUnit || 'km';
        currency = sessionSettings.currency || 'TRY';
        // Initialize tooltips
        initTooltips();

        // Load vehicles
        loadVehicles().then(() => {            
            // Initialize tables
            initTables().then(generateReport).then(setDefaultDateRange);
        });
    }
    
    async function loadVehicles() {
        http.post({
            url: '/api/vehicles/list-simple',
            loading: true
        }).then((response) => {
            if (response.success && response.data) {
                vehicleList = response.data;
                populateVehicleSelect(response.data);
            }
        }).catch((error) => {
            console.error("Error loading vehicles:", error);
        });
    }
    
    function populateVehicleSelect(vehicles) {
        const select = $('#vehicleSelect');
        select.find('option:not(:first)').remove();
        
        vehicles.forEach(vehicle => {
            select.append(`<option value="${vehicle.device_did}">${vehicle.device_name}</option>`);
        });
    }
    
    async function initTables() {
        // Initialize Vehicle Health Table
        vehicleHealthTable = createTabulator("#vehicle-health-table", {
            columns: [
                { title: "<?php echo e(__('Vehicle')); ?>", field: "vehicle_name", headerFilter: true },
                { title: "<?php echo e(__('Health Score')); ?>", field: "health_score", 
                    formatter: function(cell) {
                        const score = cell.getValue();
                        const status = getHealthStatusInfo(score);
                        return `<span class="health-score-badge health-score-${status.class}">${score}%</span>`;
                    },
                    sorter: "number"
                },
                { title: "<?php echo e(__('Status')); ?>", field: "health_status", 
                    formatter: function(cell) {
                        const status = cell.getValue();
                        const statusInfo = getHealthStatusDisplayInfo(status);
                        return `<span class="health-status-${status}"><i class="${statusInfo.icon}"></i> ${__(statusInfo.text)}</span>`;
                    }
                },
                { title: "<?php echo e(__('Next Maintenance')); ?>", field: "next_maintenance", 
                    formatter: function(cell) {
                        return cell.getValue() ? dateFormatter(cell.getValue()) : '-';
                    }
                },
                { title: "<?php echo e(__('Days to Inspection')); ?>", field: "remaining_days", 
                    formatter: function(cell) {
                        const value = cell.getValue();
                        if (value === null || value === undefined) {
                            return '<span class="text-muted">-</span>';
                        } else if (value < 0) {
                            return `<span class="text-danger">${value} <?php echo e(__('days (overdue)')); ?></span>`;
                        } else {
                            return `${value} <?php echo e(__('days')); ?>`;
                        }
                    }
                },
                
                { title: "<?php echo e(__('Maintenance Interval')); ?>", field: "maintenance_km_interval", 
                    formatter: function(cell) {
                        const value = cell.getValue();
                        return value ? `${value} ${lengthUnit}` : '-';
                    },
                    sorter: "number"
                },
                { title: "<?php echo e(__('Remaining KM')); ?>", field: "remaining_km", 
                    sorter: "number"
                },
                { title: "<?php echo e(__('Remaining KM Status')); ?>", field: "remaining_km_status", 
                    formatter: function(cell) {
                        const status = cell.getValue();
                        const statusColor = cell.getRow().getData().remaining_km_status_color;
                        return `<span class="badge text-white bg-${statusColor}">${status}</span>`;
                    }
                },
                { title: "<?php echo e(__('Driving Score')); ?>", field: "performance_score", 
                    formatter: function(cell) {
                        const score = cell.getValue();
                        return `${score}%`;
                    }
                },
                { title: "<?php echo e(__('Actions')); ?>", formatter: function(cell) {
                    const data = cell.getRow().getData();
                    return `<button class="btn btn-sm btn-primary" onclick="viewVehicleDetails('${data.did}')">
                        <i class="ti ti-eye"></i> <?php echo e(__('Details')); ?>

                    </button>`;
                }}
            ]
        });
        
    }
    
    async function generateReport() {
        const vehicleId = $('#vehicleSelect').val();
        const healthStatus = $('#healthStatusFilter').val();
        const startDate = $('#startDate').val();
        const endDate = $('#endDate').val();
        
        if (!startDate || !endDate) {
            showNotification("error", "<?php echo e(__('Please select date range')); ?>");
            return;
        }
        
        // Show loading state and hide all other content
        $('#loadingState').removeClass('d-none');
        $('#healthOverviewCards').addClass('d-none');
        $('#vehicleHealthTableCard').addClass('d-none');
        $('#healthStatusDistributionRow').addClass('d-none');
        $('#performanceAnalysisRow').addClass('d-none');
        $('#noDataState').addClass('d-none');
        $('#vehicleDetailsModal').modal('hide');
        
        const requestData = {
            vehicle_id: vehicleId,
            start_date: startDate,
            end_date: endDate
        };
        
        http.post({
            url: '/api/reports/vehicle-health',
            data: requestData,
            loading: false
        }).then((response) => {
            if (response.success && response.data) {
                reportData = response.data;

                //console.log(response.data);

                var message = __('#name# used #value# records').replace('#name#', __('Vehicle Health Report')).replace('#value#', response.data.device_logs ? response.data.device_logs.length : 0);
                showReportDataProcessingInfo(message);

                // Check if we have any vehicles to display
                const hasVehicles = response.data.vehicles && response.data.vehicles.length > 0;
                
                if (!hasVehicles) {
                    // No vehicles found - show no data state
                    $('#loadingState').addClass('d-none');
                    $('#noDataState').removeClass('d-none');
                    showNotification("warning", "<?php echo e(__('No vehicles found for the selected criteria')); ?>");
                    return;
                }

                const hasData = processRawData(response.data, healthStatus);
                
                if (hasData === false) {
                    // Data was filtered out - show no data state
                    $('#loadingState').addClass('d-none');
                    $('#noDataState').removeClass('d-none');
                    showNotification("info", "<?php echo e(__('No data found matching the selected filters')); ?>");
                    return;
                }
                
                $('#loadingState').addClass('d-none');
                $('#healthOverviewCards').removeClass('d-none');
                $('#vehicleHealthTableCard').removeClass('d-none');
                $('#healthStatusDistributionRow').removeClass('d-none');
                $('#performanceAnalysisRow').removeClass('d-none');
            } else {
                showNotification("error", response.message || "<?php echo e(__('Error generating report')); ?>");
                $('#loadingState').addClass('d-none');
                $('#noDataState').removeClass('d-none');
            }
        }).catch((error) => {
            console.error("Report generation error:", error);
            showNotification("error", "<?php echo e(__('Error generating report')); ?>");
            $('#loadingState').addClass('d-none');
            $('#noDataState').removeClass('d-none');
        });
    }
    
    function processRawData(rawData, healthStatusFilter) {

        const vehicles = rawData.vehicles || [];
        const deviceLogs = rawData.device_logs || [];
        // Use new models as flat arrays
        const alarmsInspection = rawData.alarms_inspection || [];
        const alarmsMileage = rawData.alarms_mileage || [];

        // Process vehicle health data
        const vehicleHealthData = [];
        const healthDistribution = {
            excellent: 0,
            good: 0,
            fair: 0,
            poor: 0,
            critical: 0
        };
        
        let totalHealthScore = 0;
        let totalMaintenanceDue = 0;
        let totalPerformanceScore = 0;
        
        vehicles.forEach(vehicle => {
            // Find inspection and mileage info for this vehicle
            const inspectionInfo = alarmsInspection.find(a => a.did === vehicle.did);
            const mileageInfo = alarmsMileage.find(a => a.vehicle_id === vehicle.did);
            // Calculate health metrics for this vehicle
            const vehicleLogs = deviceLogs.filter(log => log.did === vehicle.did);
            // For alarms, you may want to merge with inspectionInfo if needed
            const vehicleAlarms = []; // Not used for inspection/mileage scoring now
            const healthScoreModel = calculateVehicleHealthScore(vehicle, vehicleLogs, vehicleAlarms, inspectionInfo, mileageInfo);
            const healthScore = healthScoreModel.score;
            const maintenanceDue = getMaintenanceDue(vehicle, inspectionInfo, mileageInfo);
            const performanceScore = healthScoreModel.performanceScore;
            const lastInspection = getLastInspection(vehicle, inspectionInfo);
            const nextMaintenance = getNextMaintenance(vehicle, inspectionInfo);
            const healthStatus = getHealthStatus(healthScore);
            
            // Filter by health status if specified
            if (healthStatusFilter && healthStatusFilter !== healthStatus) {
                return;
            }
            
            let remainingDays = null;
            if (inspectionInfo && inspectionInfo.alarm_settings) {
                try {
                    const settings = JSON.parse(inspectionInfo.alarm_settings);
                    const inspectionDate = settings.inspection_date;
                    if (inspectionDate) {
                        const today = new Date();
                        const dueDate = new Date(inspectionDate);
                        remainingDays = Math.ceil((dueDate - today) / (1000 * 3600 * 24));
                    }
                } catch (e) {}
            }

            var remaining_km = (mileageInfo.km_end - mileageInfo.km_start) - mileageInfo.maintenance_km_interval;
            var remaining_km_status = "";
            var remaining_km_status_color = "";
            if (remaining_km < 0) {
                remaining_km_status = "overdue";
                remaining_km_status_color = "danger";
            } else if (remaining_km <= 500) {
                remaining_km_status = "approaching";
                remaining_km_status_color = "warning";
            } else {
                remaining_km_status = "good";
                remaining_km_status_color = "success";
            }

            vehicleHealthData.push({
                did: vehicle.did,
                vehicle_name: vehicle.name,
                health_score: healthScore,
                health_status: healthStatus,
                remaining_days: remainingDays,
                current_km: mileageInfo.km_end,
                maintenance_km_interval: mileageInfo.maintenance_km_interval,
                remaining_km: parseInt(remaining_km), 
                remaining_km_status: remaining_km_status,
                remaining_km_status_color: remaining_km_status_color,
                next_maintenance: nextMaintenance,
                performance_score: performanceScore,
                inspection_penalty: healthScoreModel.inspectionScorePenalty,
                maintenance_penalty: healthScoreModel.maintenanceScorePenalty,
                performance_score: healthScoreModel.performanceScore
            });
            
            // Update distribution
            healthDistribution[healthStatus]++;
            
            // Update totals
            totalHealthScore += healthScore;
            totalMaintenanceDue += maintenanceDue;
            totalPerformanceScore += performanceScore;
        });
        
        // Calculate overview statistics
        const vehicleCount = vehicleHealthData.length;
        
        // Check if we have any data after filtering
        if (vehicleCount === 0) {
            return false; // No data to display
        }
        
        const overview = {
            health_score: vehicleCount > 0 ? Math.round(totalHealthScore / vehicleCount * 10) / 10 : 0,
            maintenance_due: totalMaintenanceDue,
            performance_score: vehicleCount > 0 ? Math.round(totalPerformanceScore / vehicleCount * 10) / 10 : 0,
            health_breakdown: `<?php echo e(__('Average across')); ?> ${vehicleCount} <?php echo e(__('vehicles')); ?>`,
            maintenance_breakdown: "<?php echo e(__('Vehicles due for maintenance')); ?>",
            performance_breakdown: "<?php echo e(__('Average performance score')); ?>",
            health_trend: "<?php echo e(__('Last 30 days')); ?>",
            performance_trend: "<?php echo e(__('Last 30 days')); ?>"
        };
        
        // Generate maintenance schedule
        
        // Generate chart data based on actual data dates
        
        // Determine actual date range from device logs
        const dataDateRange = getDataDateRange(deviceLogs);
        
        const enginePerformance = generateEnginePerformanceData(deviceLogs, dataDateRange.start, dataDateRange.end);
        const drivingBehavior = generateDrivingBehaviorData(deviceLogs, dataDateRange.start, dataDateRange.end);
        
        // Update UI
        updateHealthOverview(overview);
        updateTables({
            vehicles: vehicleHealthData,
        });
        createCharts({
            health_distribution: healthDistribution,
            score_penalty: vehicleHealthData, 
            engine_performance: enginePerformance,
            driving_behavior: drivingBehavior
        });
        
        return true; // Data was successfully processed and displayed
    }
    
    function getDataDateRange(deviceLogs) {
        if (deviceLogs.length === 0) {
            // If no data, default to last 30 days
            const end = new Date();
            const start = new Date(end.getTime() - 30 * 24 * 60 * 60 * 1000);
            return {
                start: start.toISOString().split('T')[0],
                end: end.toISOString().split('T')[0]
            };
        }
        
        // Extract all dates from device logs
        const dates = deviceLogs.map(log => {
            const dateStr = log.dt ? log.dt.split(' ')[0] : log.dtx.split(' ')[0];
            return new Date(dateStr);
        }).filter(date => !isNaN(date.getTime()));
        
        if (dates.length === 0) {
            // If no valid dates, default to last 30 days
            const end = new Date();
            const start = new Date(end.getTime() - 30 * 24 * 60 * 60 * 1000);
            return {
                start: start.toISOString().split('T')[0],
                end: end.toISOString().split('T')[0]
            };
        }
        
        // Find min and max dates
        const minDate = new Date(Math.min(...dates.map(d => d.getTime())));
        const maxDate = new Date(Math.max(...dates.map(d => d.getTime())));
        
        // Extend range by 3 days on each side for better visualization
        const start = new Date(minDate.getTime() - 3 * 24 * 60 * 60 * 1000);
        const end = new Date(maxDate.getTime() + 3 * 24 * 60 * 60 * 1000);
        
        return {
            start: start.toISOString().split('T')[0],
            end: end.toISOString().split('T')[0]
        };
    }
    
    function calculateVehicleHealthScore(vehicle, deviceLogs, alarms, inspectionInfo, mileageInfo) {
        let score = 100;
        
        // Inspection penalty
        let inspectionScorePenalty = 0;
        let inspectionDate = null;
        if (inspectionInfo && inspectionInfo.alarm_settings) {
            try {
                const settings = JSON.parse(inspectionInfo.alarm_settings);
                inspectionDate = settings.inspection_date;
            } catch (e) {}
        }
        if (inspectionDate) {
            const today = new Date();
            const dueDate = new Date(inspectionDate);
            const daysRemaining = Math.ceil((dueDate - today) / (1000 * 3600 * 24));
            if (daysRemaining < 0) inspectionScorePenalty = 20;
            else if (daysRemaining <= 7) inspectionScorePenalty = 15;
            else if (daysRemaining <= 30) inspectionScorePenalty = 10;
            else if (daysRemaining <= 90) inspectionScorePenalty = 5;
        }
        // Maintenance penalty
        let maintenanceScorePenalty = 0;
        if (
            mileageInfo &&
            mileageInfo.maintenance_km_interval &&
            mileageInfo.km_end !== undefined &&
            mileageInfo.km_start !== undefined
        ) {
            const lastMaintenanceKm = parseFloat(mileageInfo.km_start) || 0;
            const currentKm = parseFloat(mileageInfo.km_end) || 0;
            const interval = parseInt(mileageInfo.maintenance_km_interval, 10);
            const nextMaintenanceKm = lastMaintenanceKm + interval;
            const kmToNext = nextMaintenanceKm - currentKm;
            if (kmToNext < 0) maintenanceScorePenalty = 20;
            else if (kmToNext <= 500) maintenanceScorePenalty = 15;
            else if (kmToNext <= 1000) maintenanceScorePenalty = 10;
            else if (kmToNext <= 2000) maintenanceScorePenalty = 5;
        } else {
            // Penalize missing maintenance data
            maintenanceScorePenalty = 20;
        }
        score -= inspectionScorePenalty;
        score -= maintenanceScorePenalty;
        
        // Check performance score (average with performance)
        const performanceScore = calculatePerformanceScore(deviceLogs);
        score = (score + performanceScore) / 2;

        var scoreModel = {
            score: Math.max(0, Math.min(100, Math.round(score * 10) / 10)),
            inspectionScorePenalty: inspectionScorePenalty,
            maintenanceScorePenalty: maintenanceScorePenalty,
            performanceScore: performanceScore
        };
        
        return scoreModel;
    }
        
    function calculatePerformanceScore(deviceLogs) {
        if (deviceLogs.length === 0) return 70;
        
        // Calculate average speed and other performance metrics
        const speeds = deviceLogs.map(log => log.speed || 0).filter(speed => speed > 0);
        const avgSpeed = speeds.length > 0 ? speeds.reduce((a, b) => a + b, 0) / speeds.length : 0;
        
        // Simple performance calculation based on speed consistency
        const avgSpeedSet = 80;
        let performanceScore = 70;
        if (avgSpeed > 0 && avgSpeed < avgSpeedSet) {
            performanceScore = 85 + Math.random() * 10; // Good performance
        } else if (avgSpeed >= avgSpeedSet) {
            performanceScore = 70 + Math.random() * 10; // Moderate performance
        }
        
        return Math.round(performanceScore);
    }
    
    function getLastInspection(vehicle, inspectionInfo) {
        // Since we don't keep past inspection dates, return null
        return null;
    }
    
    function getNextMaintenance(vehicle, inspectionInfo) {
        // Get inspection date from vehicle alarm settings (aid = 301)

        let inspectionDate = null;
        if (inspectionInfo && inspectionInfo.alarm_settings) {
            try {
                const settings = JSON.parse(inspectionInfo.alarm_settings);
                inspectionDate = settings.inspection_date || null;
            } catch (e) {
                console.error('Error parsing alarm settings for vehicle:', vehicle.did, e);
            }
        }
        
        return inspectionDate;
    }
    
    function getMaintenanceDue(vehicle, inspectionInfo, mileageInfo) {

        // Calculate days remaining to next inspection
        const inspectionDate = getNextMaintenance(vehicle, inspectionInfo);
        
        if (!inspectionDate) {
            return 0; // No inspection date set
        }
        
        const today = new Date();
        const dueDate = new Date(inspectionDate);
        const timeDiff = dueDate.getTime() - today.getTime();
        const daysRemaining = Math.ceil(timeDiff / (1000 * 3600 * 24));
        
        // Return 1 if overdue or due within 30 days, 0 otherwise
        if (daysRemaining < 0 || daysRemaining <= 30) {
            return 1;
        }
        
        return 0;
    }
    
    function getHealthStatus(score) {
        if (score >= 90) return 'excellent';
        if (score >= 80) return 'good';
        if (score >= 70) return 'fair';
        if (score >= 60) return 'poor';
        return 'critical';
    }
    
    function generateMaintenanceCostData(startDate, endDate) {
        const labels = [];
        const values = [];
        
        const start = new Date(startDate);
        const end = new Date(endDate);
        
        while (start <= end) {
            const dateStr = start.toISOString().split('T')[0];
            labels.push(dateStr);
            
            // Generate realistic maintenance costs based on vehicle age and usage
            // This would typically come from actual maintenance records
            const baseCost = 200;
            const randomFactor = Math.random() * 0.5 + 0.75; // 0.75 to 1.25
            const cost = Math.round(baseCost * randomFactor);
            
            values.push(cost);
            start.setDate(start.getDate() + 1);
        }
        
        return { labels, values };
    }
    
    function generateEnginePerformanceData(deviceLogs, startDate, endDate) {
        const labels = [];
        const efficiency = [];
        
        // Group logs by date
        const logsByDate = {};
        deviceLogs.forEach(log => {
            const date = log.dt ? log.dt.split(' ')[0] : log.dtx.split(' ')[0];
            if (!logsByDate[date]) {
                logsByDate[date] = [];
            }
            logsByDate[date].push(log);
        });
        
        const start = new Date(startDate);
        const end = new Date(endDate);
        
        while (start <= end) {
            const dateStr = start.toISOString().split('T')[0];
            labels.push(dateStr);
            
            const dayLogs = logsByDate[dateStr] || [];
            let dayEfficiency = 75; // Default efficiency
            
            if (dayLogs.length > 0) {
                // Calculate efficiency based on speed consistency and other factors
                const speeds = dayLogs.map(log => parseFloat(log.speed) || 0).filter(s => s > 0);
                const avgSpeed = speeds.length > 0 ? speeds.reduce((a, b) => a + b, 0) / speeds.length : 0;
                
                if (avgSpeed > 0 && avgSpeed < 120) {
                    dayEfficiency = 85 + Math.random() * 10; // Good efficiency
                } else if (avgSpeed >= 120) {
                    dayEfficiency = 70 + Math.random() * 10; // Moderate efficiency
                } else {
                    dayEfficiency = 75 + Math.random() * 10; // Default efficiency
                }
            }
            
            efficiency.push(Math.round(dayEfficiency));
            start.setDate(start.getDate() + 1);
        }
        
        return { labels, efficiency };
    }
    
    function generateDrivingBehaviorData(deviceLogs, startDate, endDate) {

        
        const labels = [];
        const harshEvents = [];
        
        // Group logs by date
        const logsByDate = {};
        deviceLogs.forEach(log => {
            const date = log.dt ? log.dt.split(' ')[0] : log.dtx.split(' ')[0];
            if (!logsByDate[date]) {
                logsByDate[date] = [];
            }
            logsByDate[date].push(log);
        });
        

        
        const start = new Date(startDate);
        const end = new Date(endDate);
        
        while (start <= end) {
            const dateStr = start.toISOString().split('T')[0];
            labels.push(dateStr);
            
            const dayLogs = logsByDate[dateStr] || [];
            let harshCount = 0;
            
            if (dayLogs.length > 0) {
                // Count harsh driving events (sudden speed changes, etc.)
                for (let i = 1; i < dayLogs.length; i++) {
                    const prevSpeed = parseFloat(dayLogs[i-1].speed) || 0;
                    const currSpeed = parseFloat(dayLogs[i].speed) || 0;
                    const speedDiff = Math.abs(currSpeed - prevSpeed);
                    
                    if (speedDiff > 20) { // Sudden speed change
                        harshCount++;
                    }
                }
            }
            
            harshEvents.push(harshCount);
            start.setDate(start.getDate() + 1);
        }
        
        const result = { labels, harsh_events: harshEvents };

        return result;
    }
    
    function updateHealthOverview(overview) {
        // Update overview cards
        $('#overallHealthScore').text(overview.health_score + '%');
        $('#maintenanceDue').text(overview.maintenance_due);
        $('#performanceScore').text(overview.performance_score + '%');
        
        // Update breakdowns
        $('#healthScoreBreakdown').text(overview.health_breakdown);
        $('#maintenanceBreakdown').text(overview.maintenance_breakdown);
        $('#performanceBreakdown').text(overview.performance_breakdown);
        
        // Update trends
        $('#healthScoreTrend').text(overview.health_trend);
        $('#performanceTrend').text(overview.performance_trend);
    }
    
    function updateTables(data) {
        // Update vehicle health table
        vehicleHealthTable.setData(data.vehicles || []);
    
    }
    
    function createCharts(data) {
        
        // Health Status Distribution Chart
        createHealthStatusChart(data.health_distribution);
        
        // Health Score Breakdown Chart
        createHealthScoreBreakdownChart(data.score_penalty);
        
        // Engine Performance Chart
        createEnginePerformanceChart(data.engine_performance);
        
        // Driving Behavior Chart
        createDrivingBehaviorChart(data.driving_behavior);
    }
    
    function createHealthStatusChart(distribution) {
        if (typeof ApexCharts === 'undefined') {
            console.error('ApexCharts is not loaded');
            return;
        }
        
        const options = {
            series: Object.values(distribution || {}),
            chart: {
                type: 'donut',
                height: 300
            },
            labels: Object.keys(distribution || {}).map(key => __(key)),
            colors: ['#28a745', '#17a2b8', '#ffc107', '#fd7e14', '#dc3545'],
            legend: {
                position: 'bottom'
            },
            plotOptions: {
                pie: {
                    donut: {
                        size: '70%'
                    }
                }
            }
        };
        
        if (window.healthStatusChart && typeof window.healthStatusChart.destroy === 'function') {
            window.healthStatusChart.destroy();
        }
        
        const chartElement = document.querySelector("#healthStatusChart");
        if (chartElement) {
            window.healthStatusChart = new ApexCharts(chartElement, options);
            window.healthStatusChart.render();
        }
    }
    
    function createHealthScoreBreakdownChart(vehicles) {

        if (typeof ApexCharts === 'undefined') {
            console.error('ApexCharts is not loaded');
            return;
        }
        const chartElement = document.querySelector('#healthScoreBreakdownChart');
        if (!chartElement) return;
        if (!vehicles || vehicles.length === 0) {
            chartElement.innerHTML = '<div class="text-center text-muted py-4">No vehicle data available</div>';
            return;
        }
        // Prepare data
        const categories = vehicles.map(v => v.vehicle_name || v.name || v.did);
        const performance = vehicles.map(v => v.performance_score !== undefined ? v.performance_score : null);
        const inspectionPenalty = vehicles.map(v => v.inspection_penalty !== undefined ? v.inspection_penalty : null);
        const maintenancePenalty = vehicles.map(v => v.maintenance_penalty !== undefined ? v.maintenance_penalty : null);
        

        const options = {
            chart: {
                type: 'bar',
                height: 350,
                stacked: false,
                toolbar: { show: false }
            },
            series: [
                {
                    name: '<?php echo e(__('Driving Score')); ?>',
                    data: performance,
                },
                {
                    name: '<?php echo e(__('Inspection Penalty')); ?>',
                    data: inspectionPenalty,
                },
                {
                    name: '<?php echo e(__('Maintenance Penalty')); ?>',
                    data: maintenancePenalty,
                },
            ],
            xaxis: {
                categories: categories,
                labels: { rotate: -45 }
            },
            yaxis: {
                title: { text: '<?php echo e(__("Value")); ?>' }
            },
            tooltip: {
                y: {
                    formatter: function(val, opts) {
                        if (opts.seriesIndex === 2) return val + '%';
                        if (opts.seriesIndex === 3) return val + '%';
                        if (opts.seriesIndex === 4) return val + '%';
                        if (opts.seriesIndex === 5) return val + '%';
                        return val;
                    }
                }
            },
            colors: ['#28a745', '#ffc107', '#007bff', '#dc3545', '#6c757d', '#17a2b8'],
            legend: { position: 'top' },
            plotOptions: {
                bar: {
                    horizontal: false,
                    dataLabels: { position: 'top' },
                    columnWidth: '50%'
                }
            },
            dataLabels: {
                enabled: true,
                formatter: function(val, opts) {
                    if (opts.seriesIndex === 2) return val + '%';
                    if (opts.seriesIndex === 3) return val + '%';
                    if (opts.seriesIndex === 4) return val + '%';
                    if (opts.seriesIndex === 5) return val + '%';
                    return val;
                }
            }
        };
        if (window.healthScoreBreakdownChart && typeof window.healthScoreBreakdownChart.destroy === 'function') {
            window.healthScoreBreakdownChart.destroy();
        }
        window.healthScoreBreakdownChart = new ApexCharts(chartElement, options);
        window.healthScoreBreakdownChart.render();
    }
    
    function createEnginePerformanceChart(performance) {
        if (typeof ApexCharts === 'undefined') {
            console.error('ApexCharts is not loaded');
            return;
        }
        
        const options = {
            series: [{
                name: '<?php echo e(__("Engine Efficiency")); ?>',
                data: performance?.efficiency || []
            }],
            chart: {
                type: 'area',
                height: 300
            },
            xaxis: {
                categories: performance?.labels || []
            },
            yaxis: {
                title: {
                    text: '<?php echo e(__("Efficiency (%)")); ?>'
                }
            },
            colors: ['#28a745']
        };
        
        if (window.enginePerformanceChart && typeof window.enginePerformanceChart.destroy === 'function') {
            window.enginePerformanceChart.destroy();
        }
        
        const chartElement = document.querySelector("#enginePerformanceChart");
        if (chartElement) {
            window.enginePerformanceChart = new ApexCharts(chartElement, options);
            window.enginePerformanceChart.render();
        }
    }
    
    function createDrivingBehaviorChart(behavior) {

        
        if (typeof ApexCharts === 'undefined') {
            console.error('ApexCharts is not loaded');
            return;
        }
        
        const chartElement = document.querySelector("#drivingBehaviorChart");
        if (!chartElement) {
            console.error('Driving behavior chart element not found');
            return;
        }
        
        // Ensure we have valid data
        const harshEvents = behavior?.harsh_events || [];
        const labels = behavior?.labels || [];
        

        
        // If no data, show a message
        if (harshEvents.length === 0 || labels.length === 0) {
            chartElement.innerHTML = '<div class="text-center text-muted py-4">No driving behavior data available</div>';
            return;
        }
        
        const options = {
            series: [{
                name: '<?php echo e(__("Harsh Driving Events")); ?>',
                data: harshEvents
            }],
            chart: {
                type: 'bar',
                height: 300,
                toolbar: {
                    show: false
                }
            },
            xaxis: {
                categories: labels,
                labels: {
                    rotate: -45,
                    rotateAlways: false
                }
            },
            yaxis: {
                title: {
                    text: '<?php echo e(__("Events Count")); ?>'
                }
            },
            colors: ['#dc3545'],
            plotOptions: {
                bar: {
                    borderRadius: 4,
                    dataLabels: {
                        position: 'top'
                    }
                }
            },
            dataLabels: {
                enabled: true,
                formatter: function (val) {
                    return val > 0 ? val : '';
                }
            }
        };
        
        if (window.drivingBehaviorChart && typeof window.drivingBehaviorChart.destroy === 'function') {
            window.drivingBehaviorChart.destroy();
        }
        
        try {
            window.drivingBehaviorChart = new ApexCharts(chartElement, options);
            window.drivingBehaviorChart.render().then(() => {
        
            }).catch((error) => {
                console.error('Error rendering driving behavior chart:', error);
            });
        } catch (error) {
            console.error('Error creating driving behavior chart:', error);
        }
    }
    
    function getHealthStatusInfo(score) {
        if (score >= 90) return { class: 'excellent', text: '<?php echo e(__("Excellent")); ?>' };
        if (score >= 80) return { class: 'good', text: '<?php echo e(__("Good")); ?>' };
        if (score >= 70) return { class: 'fair', text: '<?php echo e(__("Fair")); ?>' };
        if (score >= 60) return { class: 'poor', text: '<?php echo e(__("Poor")); ?>' };
        return { class: 'critical', text: '<?php echo e(__("Critical")); ?>' };
    }
    
    function getHealthStatusDisplayInfo(status) {
        const statusMap = {
            'excellent': { text: '<?php echo e(__("Excellent")); ?>', icon: 'ti ti-heart text-success' },
            'good': { text: '<?php echo e(__("Good")); ?>', icon: 'ti ti-heart text-info' },
            'fair': { text: '<?php echo e(__("Fair")); ?>', icon: 'ti ti-heart text-warning' },
            'poor': { text: '<?php echo e(__("Poor")); ?>', icon: 'ti ti-heart text-orange' },
            'critical': { text: '<?php echo e(__("Critical")); ?>', icon: 'ti ti-heart text-danger' }
        };
        return statusMap[status] || statusMap['fair'];
    }
    
    function viewVehicleDetails(vehicleId) {
        // Show modal with vehicle details
        $('#vehicleDetailsModal').modal('show');
        
        // Load vehicle details
        http.post({
            url: '/api/reports/vehicle-health-details',
            data: { vehicle_id: vehicleId }
        }).then((response) => {
            if (response.success && response.data) {
                const details = response.data;
                const vehicle = details.vehicle;
                const deviceLogs = details.device_logs || [];
                
                // Calculate health metrics
                const inspectionInfo = reportData.alarms_inspection.find(a => a.did === vehicle.did);
                const mileageInfo = reportData.alarms_mileage.find(a => a.vehicle_id === vehicle.did);
                const healthScoreModel = calculateVehicleHealthScore(vehicle, deviceLogs, [], inspectionInfo, mileageInfo);
                const healthScore = healthScoreModel.score;
                const healthStatus = getHealthStatus(healthScore);
                const maintenanceDue = getMaintenanceDue(vehicle, inspectionInfo, mileageInfo);
                const performanceScore = healthScoreModel.performanceScore;
                const lastInspection = getLastInspection(vehicle, inspectionInfo);
                const nextMaintenance = getNextMaintenance(vehicle, inspectionInfo);
                
                let remainingDays = null;
                if (inspectionInfo && inspectionInfo.alarm_settings) {
                    try {
                        const settings = JSON.parse(inspectionInfo.alarm_settings);
                        const inspectionDate = settings.inspection_date;
                        if (inspectionDate) {
                            const today = new Date();
                            const dueDate = new Date(inspectionDate);
                            remainingDays = Math.ceil((dueDate - today) / (1000 * 3600 * 24));
                        }
                    } catch (e) {}
                }

                // Generate HTML content
                const html = generateVehicleDetailsHTML(vehicle, healthScore, healthStatus, maintenanceDue, performanceScore, lastInspection, nextMaintenance, remainingDays);
                $('#vehicleDetailsContent').html(html);
            }
        }).catch((error) => {
            console.error("Error loading vehicle details:", error);
            $('#vehicleDetailsContent').html('<div class="alert alert-danger"><?php echo e(__("Error loading vehicle details")); ?></div>');
        });
    }
    
    function generateVehicleDetailsHTML(vehicle, healthScore, healthStatus, maintenanceDue, performanceScore, lastInspection, nextMaintenance, remainingDays) {
        const statusInfo = {
            'excellent': { color: 'success', icon: 'ti ti-heart' },
            'good': { color: 'info', icon: 'ti ti-heart' },
            'fair': { color: 'warning', icon: 'ti ti-heart' },
            'poor': { color: 'orange', icon: 'ti ti-heart' },
            'critical': { color: 'danger', icon: 'ti ti-heart' }
        };
        
        const status = statusInfo[healthStatus] || statusInfo['fair'];
        
        return `
        <div class='row'>
            <div class='col-md-6'>
                <div class='card'>
                    <div class='card-header'>
                        <h4 class='card-title'><?php echo e(__('Vehicle Information')); ?></h4>
                    </div>
                    <div class='card-body'>
                        <table class='table table-borderless'>
                            <tr>
                                <td><strong><?php echo e(__('Vehicle Name')); ?>:</strong></td>
                                <td>${vehicle.name}</td>
                            </tr>
                            <tr>
                                <td><strong><?php echo e(__('Device ID')); ?>:</strong></td>
                                <td>${vehicle.did}</td>
                            </tr>
                            <tr>
                                <td><strong><?php echo e(__('Status')); ?>:</strong></td>
                                <td><span class='badge bg-${status.color}'><i class='${status.icon}'></i> ${getHealthStatusDisplayInfo(healthStatus).text}</span></td>
                            </tr>
                        </table>
                    </div>
                </div>
            </div>
            <div class='col-md-6'>
                <div class='card'>
                    <div class='card-header'>
                        <h4 class='card-title'><?php echo e(__('Health Metrics')); ?></h4>
                    </div>
                    <div class='card-body'>
                        <div class='row text-center'>
                            <div class='col-6'>
                                <div class='h2 text-${status.color}'>${healthScore}%</div>
                                <div class='text-muted'><?php echo e(__('Health Score')); ?></div>
                            </div>
                            <div class='col-6'>
                                <div class='h2 text-info'>${performanceScore}%</div>
                                <div class='text-muted'><?php echo e(__('Driving Score')); ?></div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        <div class='row mt-3'>
            <div class='col-md-6'>
                <div class='card'>
                    <div class='card-header'>
                        <h4 class='card-title'><?php echo e(__('Maintenance Information')); ?></h4>
                    </div>
                    <div class='card-body'>
                        <table class='table table-borderless'>
                            <tr>
                                <td><strong><?php echo e(__('Last Inspection')); ?>:</strong></td>
                                <td>${lastInspection ? dateFormatter(lastInspection) : '<?php echo e(__("Not Available")); ?>'}</td>
                            </tr>
                            <tr>
                                <td><strong><?php echo e(__('Next Maintenance')); ?>:</strong></td>
                                <td>${nextMaintenance || '<?php echo e(__("Not Available")); ?>'}</td>
                            </tr>
                            <tr>
                                <td><strong><?php echo e(__('Maintenance Due')); ?>:</strong></td>
                                <td><span class='badge bg-${maintenanceDue > 0 ? 'warning' : 'success'}'>${maintenanceDue}</span></td>
                            </tr>
                            <tr>
                                <td><strong><?php echo e(__('Days to Inspection')); ?>:</strong></td>
                                <td>${remainingDays !== null && remainingDays !== undefined ? (remainingDays < 0 ? `<span class='text-danger'>${remainingDays} <?php echo e(__('days (overdue)')); ?></span>` : `${remainingDays} <?php echo e(__('days')); ?>`) : '<?php echo e(__('Not Available')); ?>'}</td>
                            </tr>
                        </table>
                    </div>
                </div>
            </div>
            <div class='col-md-6'>
                <div class='card'>
                    <div class='card-header'>
                        <h4 class='card-title'><?php echo e(__('Alerts & Issues')); ?></h4>
                    </div>
                    <div class='card-body'>
                        <table class='table table-borderless'>
                            <tr>
                                <td><strong><?php echo e(__('Recommendations')); ?>:</strong></td>
                                <td>
                                    ${generateRecommendations(healthScore, 0, maintenanceDue)}
                                </td>
                            </tr>
                        </table>
                    </div>
                </div>
            </div>
        </div>
        `;
    }
    
    function generateRecommendations(healthScore, criticalAlerts, maintenanceDue) {
        const recommendations = [];
        
        if (healthScore < 70) {
            recommendations.push("<?php echo e(__('Schedule comprehensive inspection')); ?>");
        }
        
        if (criticalAlerts > 0) {
            recommendations.push("<?php echo e(__('Address critical alerts immediately')); ?>");
        }
        
        if (maintenanceDue > 0) {
            recommendations.push("<?php echo e(__('Schedule maintenance service')); ?>");
        }
        
        if (recommendations.length === 0) {
            recommendations.push("<?php echo e(__('Vehicle is in good condition')); ?>");
        }
        
        return recommendations.join('<br>');
    }
    

    
    function setDefaultDateRange() {
        const today = new Date();
        const lastMonth = new Date(today.getFullYear(), today.getMonth() - 1, today.getDate());
        
        $('#startDate').val(lastMonth.toISOString().split('T')[0]);
        $('#endDate').val(today.toISOString().split('T')[0]);
        
        // For testing with the provided data, set to include June 2025
        // Uncomment the lines below if you want to test with the provided data
        // $('#startDate').val('2025-06-01');
        // $('#endDate').val('2025-06-30');
    }
    
    function initTooltips() {
        // Initialize Bootstrap tooltips
        const tooltipTriggerList = [].slice.call(document.querySelectorAll('[data-bs-toggle="tooltip"]'));
        tooltipTriggerList.map(function (tooltipTriggerEl) {
            return new bootstrap.Tooltip(tooltipTriggerEl);
        });
    }
    
    function goToVehicleSettings() {
        window.location.href = '/vehicle-settings';
    }
</script>
<?php $__env->stopSection(); ?>
<?php echo $__env->make('layouts.app', \Illuminate\Support\Arr::except(get_defined_vars(), ['__data', '__path']))->render(); ?><?php /**PATH C:\xampp\htdocs\aractakipv1\resources\views\reports\vehicle-health.blade.php ENDPATH**/ ?>