var menuModel = {
    actives: ["places"],
    shows: [],
};
menuSets(menuModel);

let myMap;
var placesListData = [];
var circles = {}; // Store circles for each place
var markerColor = "red"; // Default marker color

onAppReady(function () {
    http.post({
        url: "/places-list",
    }).then((response) => {
        var data = response.data;

        //console.log("places-list", data);

        myMap = new LeafletMap();

        myMap.initMap("mapPlace", {
            enablePolygonDrawing: false,
            drawingOptions: {
                position: "topleft",
                enablePolygon: false,
                enableRectangle: false,
                enableDelete: false,
            },
        });

        placesList(data);

        placesListData = data;

        //var markers = [];
        data.forEach(function (place) {
            if (place.latitude && place.longitude) {
                var marker = [
                    {
                        id: place.id,
                        device_id: 0,
                        label: place.name,
                        status: place.status_id,
                        popup: place.name,
                        coord: [
                            parseFloat(place.latitude),
                            parseFloat(place.longitude),
                        ],
                        icon: {
                            name: "fa-solid fa-location-dot",
                            color: markerColor,
                            direction: 0,
                            className: "",
                        },
                    },
                ];
                myMap.pinMarkers(marker); // add markers to the map

                if (place.radius && place.radius > 0) {
                    var circle = L.circle([place.latitude, place.longitude], {
                        radius: place.radius,
                        color: markerColor,
                        fillColor: markerColor,
                        fillOpacity: 0.2,
                    }).addTo(myMap.mapLet);

                    circles[place.id] = circle;
                }
            }
        });

        myMap.fitBounds();

        // Load and display place statistics
        updatePlaceStats(data);
    });
});

// ========================
// Tabulator for Places
// ========================

function placesList(data) {
    var table = createTabulator("#placesList", {
        buttons: [
            {
                label: __("Add New Place"),
                icon: "fas fa-plus",
                icon_color: "success",
                func: addNewPlace,
            },
        ],
        data: data,
        movableRows: true,
        rowHeader: {
            headerSort: false,
            resizable: false,
            minWidth: 30,
            width: 30,
            rowHandle: true,
            formatter: "handle",
        },
        columns: [
            {
                title: __("Name"),
                field: "name",
                editor: "input",
                minWidth: 200,
                validator: "required",
            },
            {
                title: __("Status"),
                field: "status_id",
                width: 120,
                headerSort: false,
                formatter: function (cell) {
                    var value = cell.getValue();
                    return value == 1
                        ? `<label class="form-check form-switch toggle-status-btn" 
                                title="${__("Toggle Status")}" data-status="0">
                            <input class="form-check-input bg-lime" type="checkbox" checked="checked">
                            <span class="form-check-label">
                                ${__("Active")}</span>
                        </label>`
                        : `<label class="form-check form-switch toggle-status-btn" 
                            title="${__("Toggle Status")}" data-status="1">
                            <input class="form-check-input bg-red" type="checkbox">
                            <span class="form-check-label">
                                ${__("Inactive")}</span>
                        </label>`;
                },
                cellClick: function (e, cell) {
                    e.stopPropagation();
                    e.preventDefault();

                    if (e.target.closest(".toggle-status-btn")) {
                        var newStatus = e.target
                            .closest(".toggle-status-btn")
                            .getAttribute("data-status");
                        var rowData = cell.getRow().getData();

                        clearTimeout(cell._statusToggleTimeout);
                        cell._statusToggleTimeout = setTimeout(function () {
                            togglePlaceStatus(
                                cell.getRow(),
                                rowData,
                                newStatus
                            );
                        }, 100);
                    }
                },
            },
            {
                title: __("Action"),
                width: 150,
                headerSort: false,
                formatter: function () {
                    return `
                        <div class="btn-group" role="group">
                            <button class="btn btn-sm btn-outline-warning px-2 py-1 edit-row-btn" 
                                title="${__("Edit")}">
                                <i class="fas fa-edit"></i>
                            </button>
                            <button class="btn btn-sm btn-outline-success px-2 py-1 duplicate-row-btn" 
                                title="${__("Duplicate")}">
                                <i class="fas fa-copy"></i>
                            </button>
                            <button class="btn btn-sm btn-outline-danger px-2 py-1 delete-row-btn" 
                                title="${__("Delete")}">
                                <i class="fas fa-trash"></i>
                            </button>
                            <button class="btn btn-sm btn-outline-primary px-2 py-1 map-row-btn" 
                                title="${__("Map")}">
                                <i class="fas fa-map-marker-alt"></i>
                            </button>
                        </div>
                    `;
                },
                cellClick: function (e, cell) {
                    var rowData = cell.getRow().getData();

                    if (e.target.closest(".edit-row-btn")) {
                        editPlaceRow(cell.getRow(), rowData);
                    } else if (e.target.closest(".duplicate-row-btn")) {
                        duplicatePlaceRow(rowData);
                    } else if (e.target.closest(".delete-row-btn")) {
                        deletePlaceRow(cell.getRow(), rowData);
                    } else if (e.target.closest(".map-row-btn")) {
                        mapPlaceRow(rowData);
                    }
                },
            },
        ],
    });

    table.on("rowSelected", function (row) {
        var rowData = row.getData();
        if (typeof selectPlaceOnMap === "function") {
            selectPlaceOnMap(rowData);
        }
    });

    table.on("rowDeselected", function () {
        if (typeof deselectPlaceOnMap === "function") {
            deselectPlaceOnMap();
        }
    });

    table.on("rowMoved", function (row) {
        updatePlaceOrder(table.getData());
    });

    table.on("cellEdited", function (cell) {
        var rowData = cell.getRow().getData();
        updatePlaceData(rowData);

        // Update statistics after editing
        updatePlaceStats(table.getData());
    });

    window.placesTable = table;
    return table;
}

// Function to update place statistics
function updatePlaceStats(places) {
    const totalPlaces = places.length;
    const activePlaces = places.filter((place) => place.status_id === 1).length;
    const passivePlaces = totalPlaces - activePlaces;

    // Update the display with animation
    animateCounter("#totalPlaces", totalPlaces);
    animateCounter("#activePlaces", activePlaces);
    animateCounter("#passivePlaces", passivePlaces);
}

// Function to animate counter numbers
function animateCounter(selector, finalValue) {
    const element = $(selector);
    const startValue = 0;
    const duration = 1000;
    const increment = finalValue / (duration / 16); // 60fps
    let currentValue = startValue;

    const timer = setInterval(function () {
        currentValue += increment;
        if (currentValue >= finalValue) {
            currentValue = finalValue;
            clearInterval(timer);
        }
        element.text(Math.floor(currentValue));
    }, 16);
}

// CRUD Functions

function addNewPlace() {
    var newPlace = {
        name: __("New Place"),
        status_id: 1,
        order: null,
        radius: 50,
    };

    const postModel = Base64Helper.encode(newPlace);

    http.post({
        url: "/place-add",
        data: {
            data: postModel,
        },
    })
        .then((response) => {
            if (response.success) {
                var row = window.placesTable.addRow(response.data, true);
                placesListData.push(response.data);
                showNotification("success", __("Place added successfully"));

                // Update statistics
                updatePlaceStats(window.placesTable.getData());
            } else {
                showNotification("error", __("Error adding place"));
            }
        })
        .catch((error) => {
            console.error("Add error:", error);
            showNotification("error", __("Error adding place"));
        });
}

function editPlaceRow(row, rowData) {
    var nameCell = row.getCell("name");
    if (nameCell) {
        nameCell.edit();
    }
    row.select();
}

function duplicatePlaceRow(rowData) {
    var duplicatedPlace = {
        id: rowData.id,
        name: rowData.name + " (" + __("Copy") + ")",
        status_id: rowData.status_id,
        order: null,
    };

    const postModel = Base64Helper.encode(duplicatedPlace);

    http.post({
        url: "/place-duplicate",
        data: {
            data: postModel,
        },
    })
        .then((response) => {
            if (response.success) {
                var row = window.placesTable.addRow(response.data, true);
                placesListData.push(response.data);

                // Add marker and circle to map if coordinates exist
                var place = response.data;
                if (place.latitude && place.longitude) {
                    var marker = [
                        {
                            id: place.id,
                            device_id: 0,
                            label: place.name,
                            status: place.status_id,
                            popup: place.name,
                            coord: [
                                parseFloat(place.latitude),
                                parseFloat(place.longitude),
                            ],
                            icon: {
                                name: "fa-solid fa-location-dot",
                                color: markerColor,
                                direction: 0,
                                className: "",
                            },
                        },
                    ];
                    myMap.pinMarkers(marker);

                    if (place.radius && place.radius > 0) {
                        circles[place.id] = L.circle(
                            [place.latitude, place.longitude],
                            {
                                radius: place.radius,
                                color: markerColor,
                                fillColor: markerColor,
                                fillOpacity: 0.2,
                            }
                        ).addTo(myMap.mapLet);
                    }
                }

                showNotification(
                    "success",
                    __("Place duplicated successfully")
                );

                // Update statistics
                updatePlaceStats(window.placesTable.getData());
            } else {
                showNotification("error", __("Error duplicating place"));
            }
        })
        .catch((error) => {
            console.error("Duplicate error:", error);
            showNotification("error", __("Error duplicating place"));
        });
}

function deletePlaceRow(row, rowData) {
    showConfirmDialog({
        title: __("Delete Place"),
        message: __("Are you sure you want to delete") + " : " + rowData.name,
        confirmText: __("Delete"),
        cancelText: __("Cancel"),
        type: "danger",
        onConfirm: function () {
            const postModel = Base64Helper.encode({ id: rowData.id });

            http.post({
                url: "/place-delete",
                data: {
                    data: postModel,
                },
            })
                .then((response) => {
                    if (response.success) {
                        row.delete();
                        showNotification(
                            "success",
                            __("Place deleted successfully")
                        );
                        myMap.removeMarkers(rowData.id);

                        // Remove circle if it exists
                        if (circles[rowData.id]) {
                            myMap.mapLet.removeLayer(circles[rowData.id]);
                            delete circles[rowData.id];
                        }

                        // Update statistics
                        updatePlaceStats(window.placesTable.getData());
                    } else {
                        showNotification("error", __("Error deleting place"));
                    }
                })
                .catch((error) => {
                    console.error("Delete error:", error);
                    showNotification("error", __("Error deleting place"));
                });
        },
    });
}

// Backend Communication Functions

function updatePlaceData(rowData) {
    const mappedData = {
        id: rowData.id,
        name: rowData.name,
        status_id: rowData.status_id,
        order: rowData.order,
        latitude: rowData.latitude,
        longitude: rowData.longitude,
    };

    const postModel = Base64Helper.encode(mappedData);

    http.post({
        url: "/place-edit",
        data: {
            data: postModel,
        },
    })
        .then((response) => {
            if (response.success) {
                showNotification("success", __("Place updated successfully"));
                // find marker
                var marker = placesListData.find((p) => p.id === rowData.id);
                if (marker) {
                    marker.name = rowData.name;
                    marker.status_id = rowData.status_id;
                    marker.latitude = rowData.latitude;
                    marker.longitude = rowData.longitude;

                    // remove old marker
                    myMap.removeMarkers(rowData.id);

                    // add updated marker
                    if (marker.latitude && marker.longitude) {
                        var markers = [
                            {
                                id: marker.id,
                                device_id: 0,
                                label: marker.name,
                                status: marker.status_id,
                                popup: marker.name,
                                coord: [
                                    parseFloat(marker.latitude),
                                    parseFloat(marker.longitude),
                                ],
                                icon: {
                                    name: "fa-solid fa-location-dot",
                                    color: markerColor,
                                    direction: 0,
                                    className: "",
                                },
                            },
                        ];
                        myMap.pinMarkers(markers); // add markers to the map

                        // Update circle position if it exists
                        if (circles[rowData.id]) {
                            circles[rowData.id].setLatLng([
                                parseFloat(marker.latitude),
                                parseFloat(marker.longitude),
                            ]);
                        }
                    }
                }
            } else {
                showNotification("error", __("Error updating place"));
            }
        })
        .catch((error) => {
            console.error("Update error:", error);
            showNotification("error", __("Error updating place"));
        });
}

function updatePlaceOrder(tableData) {
    var orderData = tableData.map(function (row, index) {
        return {
            id: row.id,
            order: index + 1,
        };
    });

    const postModel = Base64Helper.encode({ orders: orderData });

    http.post({
        url: "/places-update-order",
        data: {
            data: postModel,
        },
    })
        .then((response) => {
            if (response.success) {
                //console.log("Place order updated");
            }
        })
        .catch((error) => {
            console.error("Order update error:", error);
        });
}

// Map & Coordinate Editing

function mapPlaceRow(rowData) {
    // Remove all markers except the selected one
    clearOtherMarkers(rowData.id);

    // Focus map on the selected marker
    if (rowData.latitude && rowData.longitude) {
        myMap.makeMarkerDraggable(rowData.id, 1, dragedToNewPosition);

        // Remove existing circle for this place to avoid duplicates
        if (circles[rowData.id]) {
            myMap.mapLet.removeLayer(circles[rowData.id]);
            delete circles[rowData.id];
        }

        // Create fresh circle for editing
        if (rowData.radius > 0) {
            circles[rowData.id] = L.circle(
                [rowData.latitude, rowData.longitude],
                {
                    radius: rowData.radius,
                    color: markerColor,
                    fillColor: markerColor,
                    fillOpacity: 0.2,
                }
            ).addTo(myMap.mapLet);
        }

        setTimeout(() => {
            myMap.focusMarker(rowData.id);
        }, (timeout = 500));
    }

    // Show coordinate editing interface
    showCoordinateEditingInterface(rowData);
}

function clearOtherMarkers(keepMarkerId) {
    placesListData.forEach((marker) => {
        if (marker.id !== keepMarkerId) {
            myMap.removeMarkers(marker.id);

            if (circles[marker.id]) {
                myMap.mapLet.removeLayer(circles[marker.id]);
                delete circles[marker.id];
            }
        }
    });
}

function showCoordinateEditingInterface(rowData) {
    removeCoordinateEditingInterface();

    var editingInterface = $(`
        <div class="btn-group w-100">
            <button class="btn btn-primary btn-square mb-0" onclick="exitCoordinateEditingMode()">
                <i class="fas fa-arrow-left me-3"></i> ${__("Back to List")}
            </button>
        </div>
        <div id="coordinate-editing-interface" class="p-3">
            <input type="hidden" 
                id="place-coordinates-${rowData.id}" 
                name="place_coordinates[${rowData.id}]" 
                value='${JSON.stringify([
                    rowData.latitude,
                    rowData.longitude,
                ])}'>
            <h3 class="mb-3">${rowData.name}</h3>
            <div class="mb-3">
                <div class="alert alert-warning alert-sm mb-2">
                    <i class="fas fa-exclamation-triangle"></i>
                    ${__(
                        "Drag marker or click on the map to set new coordinates for this place."
                    )}
                </div>
            </div>
            <div id="coordinate-save-options-${rowData.id}">
                <div class="mb-3">
                    <label for="radiusInput" class="form-label">
                        ${__("Radius")}
                    </label>
                    <div class="d-flex align-items-center gap-2">
                        <input type="number" 
                            id="radiusInput-${rowData.id}" 
                            min="0" max="1000" step="1" 
                            value="${rowData.radius}" 
                            class="form-control" style="width:auto !important">
                        <input type="range" 
                            id="radiusSlider-${rowData.id}" 
                            min="0" max="1000" step="10" 
                            value="${rowData.radius}" class="form-range">
                    </div>
                </div>
                <div class="btn-group w-100 mb-2">
                    <button class="btn btn-success save-coordinates-btn" 
                        data-place-id="${rowData.id}">
                        <i class="fas fa-save"></i> ${__("Save")}
                    </button>
                    <button class="btn btn-secondary cancel-coordinates-btn" 
                        data-place-id="${rowData.id}">
                        <i class="fas fa-times"></i> ${__("Cancel")}
                    </button>
                </div>
            </div>
        </div>
    `);

    $("#placeForm").html(editingInterface);
    $("#placeForm").show();
    $("#placesList").hide();

    $("#radiusInput-" + rowData.id).on("input", function () {
        var val = parseInt($(this).val()) || 0;
        $("#radiusSlider-" + rowData.id).val(val);
        circles[rowData.id].setRadius(val);
    });

    $("#radiusSlider-" + rowData.id).on("input", function () {
        var val = parseInt($(this).val()) || 0;
        $("#radiusInput-" + rowData.id).val(val);
        circles[rowData.id].setRadius(val);
    });

    setupCoordinateEditingEvents(rowData.id);

    console.log("showCoordinateEditingInterface:", rowData);

    // Map click event for setting coordinates
    myMap.mapLet.on("click", function onMapClick(e) {
        var lat = e.latlng.lat;
        var lng = e.latlng.lng;

        console.log("Map clicked at:", rowData.id, ": ", lat, lng);

        // Update marker on map
        myMap.removeMarkers(rowData.id);

        var place = placesListData.find((p) => p.id === rowData.id);
        console.log("place:", place);
        var markers = [];
        markers.push({
            id: place.id,
            device_id: 0,
            draggable: true,
            func_onDragEnd: dragedToNewPosition,
            label: place.name,
            status: place.status_id,
            popup: place.name,
            coord: [lat, lng],
            icon: {
                name: "fa-solid fa-location-dot",
                color: markerColor,
                direction: 0,
                className: "",
            },
        });

        // Update or create circle
        if (circles[place.id]) {
            // Update existing circle position
            circles[place.id].setLatLng([lat, lng]);
        } else {
            // Create new circle only if it doesn't exist
            circles[place.id] = L.circle([lat, lng], {
                radius: place.radius || 50,
                color: markerColor,
                fillColor: markerColor,
                fillOpacity: 0.2,
            }).addTo(myMap.mapLet);
        }

        myMap.pinMarkers(markers); // add markers to the map

        // Update hidden input
        var hiddenInput = document.getElementById(
            "place-coordinates-" + rowData.id
        );
        if (hiddenInput) {
            hiddenInput.value = JSON.stringify([lat, lng]);
        }

        // Show save/cancel options
        //$(`#coordinate-save-options-${rowData.id}`).show();

        // Remove this click event after one use
        myMap.mapLet.off("click", onMapClick);
    });
}

function dragedToNewPosition(id, latlng) {
    var hiddenInput = document.getElementById("place-coordinates-" + id);
    if (hiddenInput) {
        hiddenInput.value = JSON.stringify([latlng.lat, latlng.lng]);
    }

    if (circles[id]) {
        // Update circle position if it exists
        circles[id].setLatLng(latlng);
    }

    // Show save/cancel options
    //$(`#coordinate-save-options-${id}`).show();
}

function setupCoordinateEditingEvents(placeId) {
    $(document).on("click", ".save-coordinates-btn", function () {
        var placeId = $(this).data("place-id");
        saveCoordinatesChanges(placeId);
    });

    $(document).on("click", ".cancel-coordinates-btn", function () {
        var placeId = $(this).data("place-id");
        cancelCoordinatesChanges(placeId);
    });
}

function saveCoordinatesChanges(placeId) {
    var hiddenInput = document.getElementById("place-coordinates-" + placeId);

    try {
        var coordinates = [];
        if (hiddenInput && hiddenInput.value) {
            coordinates = JSON.parse(hiddenInput.value);
        }

        var postData = {
            id: placeId,
            latitude: coordinates[0],
            longitude: coordinates[1],
            radius: parseInt($(`#radiusInput-${placeId}`).val()) || 0,
        };

        const postModel = Base64Helper.encode(postData);

        $(".save-coordinates-btn")
            .prop("disabled", true)
            .html('<i class="fas fa-spinner fa-spin"></i> ' + __("Saving..."));

        http.post({
            url: "/place-save-coordinates",
            data: {
                data: postModel,
            },
        })
            .then((response) => {
                if (response.success) {
                    showNotification(
                        "success",
                        __("Coordinates updated successfully")
                    );

                    // Update table data
                    if (window.placesTable) {
                        window.placesTable.updateData([
                            {
                                id: placeId,
                                latitude: coordinates[0],
                                longitude: coordinates[1],
                                radius: postData.radius,
                            },
                        ]);
                    }

                    // Update placesListData array
                    var placeIndex = placesListData.findIndex(
                        (p) => p.id === placeId
                    );
                    if (placeIndex !== -1) {
                        placesListData[placeIndex].latitude = coordinates[0];
                        placesListData[placeIndex].longitude = coordinates[1];
                        placesListData[placeIndex].radius = postData.radius;
                    }

                    //$(`#coordinate-save-options-${placeId}`).hide();
                } else {
                    showNotification("error", __("Error updating coordinates"));
                }
            })
            .catch((error) => {
                console.error("Save coordinates error:", error);
                showNotification("error", __("Error updating coordinates"));
            })
            .finally(() => {
                $(".save-coordinates-btn")
                    .prop("disabled", false)
                    .html('<i class="fas fa-save"></i> ' + __("Save"));
            });
    } catch (error) {
        console.error("Invalid coordinate data:", error);
        showNotification("error", __("Invalid coordinate data"));
    }
}

function cancelCoordinatesChanges(placeId) {
    // Find the original place data from the loaded list
    var place = placesListData.find((p) => p.id === placeId);
    if (place) {
        // Remove the current marker for this place
        myMap.removeMarkers(placeId);

        // If original coordinates exist, re-add the marker at the original position
        if (place.latitude && place.longitude) {
            var markers = [
                {
                    id: place.id,
                    device_id: 0,
                    label: place.name,
                    status: place.status_id,
                    popup: place.name,
                    //draggable: true,
                    //func_onDragEnd: dragedToNewPosition,
                    coord: [
                        parseFloat(place.latitude),
                        parseFloat(place.longitude),
                    ],
                    icon: {
                        name: "fa-solid fa-location-dot",
                        color: markerColor,
                        direction: 0,
                        className: "",
                    },
                },
            ];
            myMap.pinMarkers(markers);

            // Update circle to original position and radius
            if (place.radius && place.radius > 0) {
                if (circles[place.id]) {
                    // Update existing circle
                    circles[place.id].setLatLng([
                        place.latitude,
                        place.longitude,
                    ]);
                    circles[place.id].setRadius(place.radius);
                } else {
                    // Create new circle if it doesn't exist
                    circles[place.id] = L.circle(
                        [place.latitude, place.longitude],
                        {
                            radius: place.radius,
                            color: markerColor,
                            fillColor: markerColor,
                            fillOpacity: 0.2,
                        }
                    ).addTo(myMap.mapLet);
                }
            } else if (circles[place.id]) {
                // Remove circle if radius is 0
                myMap.mapLet.removeLayer(circles[place.id]);
                delete circles[place.id];
            }
        }

        // Update the hidden input with the original coordinates
        var hiddenInput = document.getElementById(
            "place-coordinates-" + placeId
        );
        if (hiddenInput) {
            hiddenInput.value = JSON.stringify([
                place.latitude,
                place.longitude,
            ]);
        }

        // Hide the save/cancel options
        //$(`#coordinate-save-options-${placeId}`).hide();

        showNotification("info", __("Changes cancelled"));
    }
}

function exitCoordinateEditingMode() {
    removeCoordinateEditingInterface();

    $("#placeForm").hide();
    $("#placesList").show();

    if (myMap.drawControl) {
        myMap.mapLet.removeControl(myMap.drawControl);
        myMap.drawControl = null;
    }

    reloadAllMarkers();
}

function removeCoordinateEditingInterface() {
    $("#coordinate-editing-interface").remove();
    $(document).off("click", ".save-coordinates-btn");
    $(document).off("click", ".cancel-coordinates-btn");
}

function reloadAllMarkers() {
    http.post({
        url: "/places-list",
    }).then((response) => {
        var data = response.data;

        // Remove all markers
        myMap.removeMarkers();

        // Remove all circles from map and clear the circles object
        Object.keys(circles).forEach(function (id) {
            if (circles[id]) {
                myMap.mapLet.removeLayer(circles[id]);
            }
        });
        circles = {}; // Clear the circles object

        data.forEach(function (place) {
            if (place.latitude && place.longitude) {
                var marker = [
                    {
                        id: place.id,
                        device_id: 0,
                        label: place.name,
                        status: place.status_id,
                        popup: place.name,
                        coord: [
                            parseFloat(place.latitude),
                            parseFloat(place.longitude),
                        ],
                        icon: {
                            name: "fa-solid fa-location-dot",
                            color: markerColor,
                            direction: 0,
                            className: "",
                        },
                    },
                ];
                myMap.pinMarkers(marker); // add markers to the map

                // Create new circles
                if (place.radius && place.radius > 0) {
                    circles[place.id] = L.circle(
                        [place.latitude, place.longitude],
                        {
                            radius: place.radius,
                            color: markerColor,
                            fillColor: markerColor,
                            fillOpacity: 0.2,
                        }
                    ).addTo(myMap.mapLet);
                }
            }
        });

        myMap.fitBounds();
    });
}

// Export functions to global scope
window.addNewPlace = addNewPlace;
window.duplicatePlaceRow = duplicatePlaceRow;
window.editPlaceRow = editPlaceRow;
window.deletePlaceRow = deletePlaceRow;
window.togglePlaceStatus = togglePlaceStatus;
window.refreshPlacesTable = function () {
    http.post({
        url: "/places-list",
    }).then((response) => {
        window.placesTable.setData(response.data);
        // Update statistics with new data
        updatePlaceStats(response.data);
    });
};
window.mapPlaceRow = mapPlaceRow;
window.exitCoordinateEditingMode = exitCoordinateEditingMode;
window.saveCoordinatesChanges = saveCoordinatesChanges;
window.cancelCoordinatesChanges = cancelCoordinatesChanges;

// Toggle status function
function togglePlaceStatus(row, rowData, newStatus) {
    const postModel = Base64Helper.encode({
        id: rowData.id,
        status_id: newStatus,
    });

    http.post({
        url: "/place-toggle-status",
        data: {
            data: postModel,
        },
    })
        .then((response) => {
            if (response.success) {
                row.update({ status_id: parseInt(newStatus) });
                showNotification("success", __("Status updated successfully"));

                // Update statistics
                updatePlaceStats(window.placesTable.getData());
            } else {
                showNotification("error", __("Error updating status"));
            }
        })
        .catch((error) => {
            console.error("Toggle status error:", error);
            showNotification("error", __("Error updating status"));
        });
}

// Function to refresh statistics
window.refreshPlaceStats = function () {
    if (window.placesTable) {
        updatePlaceStats(window.placesTable.getData());
    }
};
