// --------------------- APP INITIALIZATION ---------------------
// Helper function for page scripts to wait for initialization
function onAppReady(callback) {
    http.hideLoading();
    $.when(window.appInitialized).then(callback);
}
// ******************* END OF APP INITIALIZATION *******************
// --------------------- HTTP METHODS ---------------------
// GET: data read/view ... can be cached
// POST: data create ... cannot be cached
// PUT: data update (full update) ... cannot be cached
// DELETE: data delete ... cannot be cached
// PATCH: data partial update (partial update) ... cannot be cached
const http = {
    showLoading() {
        $("#loading").removeClass("d-none");
    },
    hideLoading() {
        $("#loading").addClass("d-none");
    },

    // Common AJAX function
    _makeRequest(method, model) {
        var url = model.url;
        var data = model.data || {};
        var loading = model.loading == undefined ? true : model.loading;
        var contentType =
            model.contentType ||
            "application/x-www-form-urlencoded; charset=UTF-8";

        if (loading) {
            this.showLoading();
        }

        return new Promise((resolve, reject) => {
            $.ajax({
                url: url,
                type: method,
                data: data,
                contentType: contentType,
                headers: {
                    "X-CSRF-TOKEN": $('meta[name="csrf-token"]').attr(
                        "content"
                    ),
                },
                success: (res) => {
                    if (loading) {
                        this.hideLoading();
                    }
                    resolve(res);
                },
                error: (xhr) => {
                    if (loading) {
                        this.hideLoading();
                    }

                    // Try to parse JSON response
                    let errorResponse = xhr.responseText || xhr.statusText;
                    try {
                        const parsed = JSON.parse(xhr.responseText);
                        if (parsed) {
                            errorResponse = parsed;
                        }
                    } catch (e) {
                        // If parsing fails, keep as string
                    }

                    reject(errorResponse);
                },
            });
        });
    },

    get(model) {
        return this._makeRequest("GET", model);
    },

    post(model) {
        return this._makeRequest("POST", model);
    },

    put(model) {
        return this._makeRequest("PUT", model);
    },

    patch(model) {
        return this._makeRequest("PATCH", model);
    },

    delete(model) {
        return this._makeRequest("DELETE", model);
    },

    // Laravel special method spoofing function
    // HTML forms only support GET and POST
    putForm(model) {
        if (!model.data) model.data = {};
        model.data._method = "PUT";
        return this.post(model);
    },

    patchForm(model) {
        if (!model.data) model.data = {};
        model.data._method = "PATCH";
        return this.post(model);
    },

    deleteForm(model) {
        if (!model.data) model.data = {};
        model.data._method = "DELETE";
        return this.post(model);
    },

    // JSON API için özel fonksiyonlar
    jsonGet(model) {
        model.contentType = "application/json";
        return this.get(model);
    },

    jsonPost(model) {
        model.contentType = "application/json";
        if (model.data) {
            model.data = JSON.stringify(model.data);
        }
        return this.post(model);
    },

    jsonPut(model) {
        model.contentType = "application/json";
        if (model.data) {
            model.data = JSON.stringify(model.data);
        }
        return this.put(model);
    },

    jsonPatch(model) {
        model.contentType = "application/json";
        if (model.data) {
            model.data = JSON.stringify(model.data);
        }
        return this.patch(model);
    },

    jsonDelete(model) {
        model.contentType = "application/json";
        if (model.data) {
            model.data = JSON.stringify(model.data);
        }
        return this.delete(model);
    },
};
// ******************* END OF HTTP *******************

// --------------------- TRANSLATIONS ---------------------
var lang = $("html").attr("lang");
function __(key) {
    //console.log("Çeviri yapılıyor:", key);
    if (lang == "en") {
        return key;
    }
    return window.translations && window.translations[key]
        ? window.translations[key]
        : key;
}
// ******************* END OF TRANSLATIONS *******************

// --------------------- SESSION SETTINGS ---------------------

// HOW TO USE
// sessionSettings.locale
// sessionSettings.timezone
// sessionSettings.dateFormat
// sessionSettings.timeFormat
// sessionSettings.roles
// sessionSettings.userId
// sessionSettings.currency
// sessionSettings.speedUnit

// Global initialization promise
window.appInitialized = $.Deferred();

$(document).ready(function () {
    // Load session settings when page loads
    loadSessionSettings();
});

function loadSessionSettings() {
    // Check if settings exist in localStorage
    const cachedSettings = localStorage.getItem("app_settings");

    if (cachedSettings) {
        // If saved, get from localStorage
        try {
            window.sessionSettings = JSON.parse(cachedSettings);

            // Set locale cookie for backend to use
            if (window.sessionSettings.locale) {
                document.cookie = `app_locale=${window.sessionSettings.locale}; path=/; max-age=31536000`;

                // Check if current page locale matches localStorage locale
                const currentLocale = $("html").attr("lang");
                const localeReloaded =
                    sessionStorage.getItem("locale_reloaded");

                if (
                    currentLocale !== window.sessionSettings.locale &&
                    !localeReloaded
                ) {
                    // Mark as reloaded to prevent infinite loop
                    sessionStorage.setItem("locale_reloaded", "1");
                    // Reload page to apply new locale
                    console.log(
                        `Locale mismatch detected. Reloading page with locale: ${window.sessionSettings.locale}`
                    );
                    window.location.reload();
                    return;
                } else if (
                    currentLocale === window.sessionSettings.locale &&
                    localeReloaded
                ) {
                    // Clear the reload flag when locale matches
                    sessionStorage.removeItem("locale_reloaded");
                }
            }

            window.appInitialized.resolve();
        } catch (e) {
            console.warn(
                "Failed to parse cached settings, loading from server"
            );
            loadFromServer();
        }
    } else {
        // If not saved, get from server and save to localStorage
        loadFromServer();
    }
}

function loadFromServer() {
    http.post({ url: "/session-settings", data: {} })
        .then((response) => {
            if (response.success && response.data) {
                sessionSettings = response.data;
                window.sessionSettings = response.data;

                // Save to localStorage for next time
                localStorage.setItem(
                    "app_settings",
                    JSON.stringify(response.data)
                );

                // Set locale cookie for backend to use
                if (response.data.locale) {
                    document.cookie = `app_locale=${response.data.locale}; path=/; max-age=31536000`;

                    // Check if current page locale matches server locale
                    const currentLocale = $("html").attr("lang");
                    const localeReloaded =
                        sessionStorage.getItem("locale_reloaded");

                    if (
                        currentLocale !== response.data.locale &&
                        !localeReloaded
                    ) {
                        // Mark as reloaded to prevent infinite loop
                        sessionStorage.setItem("locale_reloaded", "1");
                        // Reload page to apply new locale
                        console.log(
                            `Locale mismatch detected. Reloading page with locale: ${response.data.locale}`
                        );
                        window.location.reload();
                        return;
                    } else if (
                        currentLocale === response.data.locale &&
                        localeReloaded
                    ) {
                        // Clear the reload flag when locale matches
                        sessionStorage.removeItem("locale_reloaded");
                    }
                }

                window.appInitialized.resolve();
            }
        })
        .catch((error) => {
            console.error("Error loading session settings:", error);
            window.appInitialized.resolve();
        });
}

// ******************* END OF SESSION SETTINGS *******************

// --------------------- MENU SETS ---------------------
function menuSets(menuModel) {
    $(`.menu-item`).removeClass("active");
    $(`.menu-item`).removeClass("show");

    menuModel.actives.forEach((id) => {
        $(`#${id}`).addClass("active");
    });
    menuModel.shows.forEach((id) => {
        $(`#${id}`).addClass("show");
    });
}

// ******************* END OF MENU SETS *******************

// --------------------- DATE TIME FORMATER ---------------------
function dateTimeFormatter(date, show_seconds = true) {
    // Handle null, undefined, or invalid input
    if (!date) {
        return "";
    }

    // date format should be YYYY-MM-DD HH:mm:ss
    const d = new Date(date);

    // Check if date is valid
    if (isNaN(d.getTime())) {
        return "";
    }

    // Check if sessionSettings is available, use defaults if not
    if (!window.sessionSettings) {
        console.warn("sessionSettings not available, using defaults");
        var format = "d.m.y";
        var timeFormat = "H:i:s";
    } else {
        var format = sessionSettings.dateFormat || "d.m.y";
        var timeFormat = sessionSettings.timeFormat || "H:i:s";
    }

    const TIME_FORMAT_MAP = {
        "24h_colon": "H:m:s",
        "24h_double": "HH:mm:ss",
        "12h_colon": "h:m:s A",
        "12h_double": "hh:mm:ss A",
    };

    // Map time format if it's in the mapping, otherwise use as is
    if (TIME_FORMAT_MAP[timeFormat]) {
        timeFormat = TIME_FORMAT_MAP[timeFormat];
    }

    // Get time format from session settings
    //let timeFormat = sessionSettings?.timeFormat || "H:i:s";
    if (show_seconds == false) {
        timeFormat = timeFormat.replace(":s", "");
    }
    // Map PHP/JS tokens to actual values
    const pad = (num) => num.toString().padStart(2, "0");
    let hours24 = d.getHours();
    let hours12 = hours24 % 12;
    hours12 = hours12 ? hours12 : 12; // 0 should be 12
    const ampm = hours24 >= 12 ? "PM" : "AM";

    // Build replacements for both PHP and JS style tokens
    const replacements = {
        // Date
        YYYY: d.getFullYear(),
        YY: d.getFullYear().toString().slice(-2),
        MM: pad(d.getMonth() + 1),
        M: d.getMonth() + 1,
        DD: pad(d.getDate()),
        D: d.getDate(),
        // Time
        HH: pad(hours24), // 24-hour, zero-padded
        H: hours24, // 24-hour, no padding
        hh: pad(hours12), // 12-hour, zero-padded
        h: hours12, // 12-hour, no padding
        mm: pad(d.getMinutes()),
        m: d.getMinutes(),
        ss: show_seconds ? pad(d.getSeconds()) : "",
        s: show_seconds ? d.getSeconds() : "",
        A: ampm,
        a: ampm.toLowerCase(),
        // PHP style
        i: pad(d.getMinutes()),
        s: show_seconds ? pad(d.getSeconds()) : "",
    };

    // Determine date part
    let datePart;
    switch (format) {
        case "d.m.y":
            datePart = "DD.MM.YYYY";
            break;
        case "y-m-d":
            datePart = "YYYY-MM-DD";
            break;
        case "m/d/y":
            datePart = "MM/DD/YYYY";
            break;
        case "d/m/y":
            datePart = "DD/MM/YYYY";
            break;
        default:
            datePart = "DD.MM.YYYY";
            break;
    }

    // Combine with time format
    let fullFormat = datePart + " " + timeFormat;

    // Replace all tokens (longest first to avoid partial replacement)
    const tokenRegex = /YYYY|YY|MM|M|DD|D|HH|H|hh|h|mm|m|ss|s|A|a|i/g;
    fullFormat = fullFormat.replace(tokenRegex, (match) =>
        replacements[match] !== undefined ? replacements[match] : match
    );

    return fullFormat.trim();
}
// ******************* END OF DATE TIME FORMATER *******************
function dateFormatter(date) {
    // Handle null, undefined, or invalid input
    if (!date) {
        return "";
    }

    // date format should be YYYY-MM-DD or similar format
    if (typeof date === "string" && date.includes(" ")) {
        date = date.split(" ")[0];
    }

    const d = new Date(date);

    // Check if date is valid
    if (isNaN(d.getTime())) {
        return "";
    }

    // Check if sessionSettings is available
    if (!sessionSettings || !sessionSettings.dateFormat) {
        return d.toISOString().split("T")[0]; // Default to YYYY-MM-DD
    }

    var format = sessionSettings.dateFormat;

    // Define pad function locally
    const pad = (num) => num.toString().padStart(2, "0");

    // Build replacements for both PHP and JS style tokens
    const replacements = {
        // Date
        YYYY: d.getFullYear(),
        YY: d.getFullYear().toString().slice(-2),
        MM: pad(d.getMonth() + 1),
        M: d.getMonth() + 1,
        DD: pad(d.getDate()),
        D: d.getDate(),
    };

    // Determine date part
    let datePart;
    switch (format) {
        case "d.m.y":
            datePart = "DD.MM.YYYY";
            break;
        case "y-m-d":
            datePart = "YYYY-MM-DD";
            break;
        case "m/d/y":
            datePart = "MM/DD/YYYY";
            break;
        case "d/m/y":
            datePart = "DD/MM/YYYY";
            break;
        default:
            datePart = "DD.MM.YYYY";
            break;
    }

    // Replace all tokens (longest first to avoid partial replacement)
    const tokenRegex = /YYYY|YY|MM|M|DD|D/g;
    const fullFormat = datePart.replace(tokenRegex, (match) =>
        replacements[match] !== undefined ? replacements[match] : match
    );

    return fullFormat.trim();
}
// --------------------- BASE64 HELPER ---------------------
const Base64Helper = {
    encode: function (model) {
        try {
            const json = JSON.stringify(model);
            return btoa(unescape(encodeURIComponent(json)));
        } catch (e) {
            //console.error("Model encode error:", e);
            return null;
        }
    },

    decode: function (encoded) {
        try {
            const json = decodeURIComponent(escape(atob(encoded)));
            return JSON.parse(json); // Geri model (obje/array) olarak döner
        } catch (e) {
            //console.error("Model decode error:", e);
            return null;
        }
    },
};
// ******************* END OF BASE64 HELPER *******************

// --------------------- CONFIRM DIALOG ---------------------
function showConfirmDialog(options) {
    const defaults = {
        title: "Confirm",
        message: "Are you sure?",
        confirmText: "Yes",
        cancelText: "Cancel",
        type: "primary", // primary, danger, warning, success
        showCancelButton: true,
        showConfirmButton: true,
        onConfirm: function () {},
        onCancel: function () {},
    };

    const config = Object.assign({}, defaults, options);

    // Modal backdrop
    const backdrop = document.createElement("div");
    backdrop.className = "modal-backdrop fade show";
    backdrop.style.zIndex = "10000";

    // Modal container
    const modal = document.createElement("div");
    modal.className = "modal fade show";
    modal.style.cssText = "display: block; z-index: 10001;";
    modal.setAttribute("tabindex", "-1");

    // Icon ve color ayarları
    let iconClass, buttonClass;
    switch (config.type) {
        case "danger":
            iconClass = "fas fa-exclamation-triangle text-danger";
            buttonClass = "btn-danger";
            break;
        case "warning":
            iconClass = "fas fa-exclamation-triangle text-warning";
            buttonClass = "btn-warning";
            break;
        case "success":
            iconClass = "fas fa-check-circle text-success";
            buttonClass = "btn-success";
            break;
        default:
            iconClass = "fas fa-question-circle text-primary";
            buttonClass = "btn-primary";
    }

    modal.innerHTML = `
        <div class="modal-dialog modal-dialog-centered">
            <div class="modal-content">
                <div class="modal-header border-0 pb-0">
                    <h5 class="modal-title d-flex align-items-center">
                        <i class="${iconClass}" style="font-size: 24px; margin-right: 10px;"></i>
                        ${config.title}
                    </h5>
                </div>
                <div class="modal-body pt-2">
                    <p class="mb-0" style="font-size: 16px; line-height: 1.5;">
                        ${config.message}
                    </p>
                </div>
                <div class="modal-footer border-0 pt-2">
                    <button type="button" class="btn btn-secondary cancel-btn ${
                        config.showCancelButton ? "" : "d-none"
                    }" style="min-width: 80px;">
                        <i class="fas fa-times me-2"></i> ${config.cancelText}
                    </button>
                    <button type="button" class="btn ${buttonClass} confirm-btn 
                    ${
                        config.showConfirmButton ? "" : "d-none"
                    } " style="min-width: 80px;">
                        <i class="fas fa-check me-2"></i> ${config.confirmText}
                    </button>
                </div>
            </div>
        </div>
    `;

    // Event listeners
    const confirmBtn = modal.querySelector(".confirm-btn");
    const cancelBtn = modal.querySelector(".cancel-btn");

    function closeModal() {
        backdrop.classList.remove("show");
        modal.classList.remove("show");
        setTimeout(() => {
            if (backdrop.parentNode) backdrop.parentNode.removeChild(backdrop);
            if (modal.parentNode) modal.parentNode.removeChild(modal);
        }, 150);
    }

    confirmBtn.addEventListener("click", function () {
        closeModal();
        config.onConfirm();
    });

    cancelBtn.addEventListener("click", function () {
        closeModal();
        config.onCancel();
    });

    // ESC tuşu ile kapatma
    function handleKeyDown(e) {
        if (e.key === "Escape") {
            closeModal();
            config.onCancel();
            document.removeEventListener("keydown", handleKeyDown);
        }
    }
    document.addEventListener("keydown", handleKeyDown);

    // Backdrop tıklama ile kapatma
    backdrop.addEventListener("click", function () {
        closeModal();
        config.onCancel();
    });

    // DOM'a ekle
    document.body.appendChild(backdrop);
    document.body.appendChild(modal);

    // Focus confirm button
    setTimeout(() => {
        confirmBtn.focus();
    }, 100);
}
// ******************* END OF CONFIRM DIALOG *******************

// --------------------- TABULATOR LOCALIZE ---------------------
function tabulatorLocalize() {
    return {
        default: {
            pagination: {
                page_size: __("Page Size"),
                page_title: __("Show Page Number"),
                first: __("First"),
                first_title: __("First Page"),
                last: __("Last"),
                last_title: __("Last Page"),
                prev: __("Previous"),
                prev_title: __("Previous Page"),
                next: __("Next"),
                next_title: __("Next Page"),
                all: __("All"),
                counter: {
                    showing: __("Showing"),
                    of: __("of"),
                    rows: __("rows"),
                    pages: __("pages"),
                },
            },
            data: {
                loading: __("Loading..."),
                error: __("Error"),
            },
            header: {
                sort_asc: __("Sort Ascending"),
                sort_desc: __("Sort Descending"),
            },
        },
    };
}
// --------------------- END OF TABULATOR LOCALIZE *******************

// --------------------- NOTIFICATION ---------------------
function showNotification(type, message) {
    if (typeof toastr !== "undefined") {
        toastr[type](message);
    } else {
        alert(message);
    }
}
// ******************* END OF NOTIFICATION *******************

// --------------------- POPOVER ---------------------

function popoverInit() {
    // Reinitialize popovers for visible items
    $('[data-bs-toggle="popover"]').popover("dispose");
    $('[data-bs-toggle="popover"]').popover();
}
function popoverDestroy() {
    $('[data-bs-toggle="popover"]').popover("dispose");
}
function popoverShow(divId) {
    $(`#${divId}`).popover("show");
}
function popoverHide(divId) {
    $(`#${divId}`).popover("hide");
}

function popoverAdd(settings) {
    // default settings if not provided
    if (!settings.trigger) {
        settings.trigger = "hover";
    }
    if (!settings.content) {
        settings.content = "";
    }
    if (!settings.placement) {
        settings.placement = "top";
    }
    $(`#${settings.divId}`).popover("dispose");
    $(`#${settings.divId}`).popover({
        content: settings.content,
        trigger: settings.trigger,
        placement: settings.placement,
    });
}

// ******************* END OF POPOVER *******************

// --------------------- COLOUR GENERATION ---------------------
function generateDistinctColors(count) {
    const colors = [];
    const hueStep = 360 / count;
    const saturation = 85; // High saturation for distinct colors
    const lightness = 55; // Medium lightness for good visibility
    for (let i = 0; i < count; i++) {
        const hue = (i * hueStep) % 360;
        const color = `hsl(${hue}, ${saturation}%, ${lightness}%)`;
        colors.push(color);
    }
    return colors;
}
// ******************* END OF COLOUR GENERATION *******************

// --------------------- SIMPLE TABULATOR WRAPPER ---------------------
/**
 * Simple centralized tabulator with standard configuration
 * @param {string} elementId - The ID of the table element
 * @param {object} options - Configuration options
 * @returns {object} Tabulator instance
 */
function createTabulator(elementId, options = {}) {
    // Standard configuration that applies to all tables
    const standardConfig = {
        langs: tabulatorLocalize(),
        layout: "fitColumns",
        renderHorizontal: "virtual",
        printAsHtml: true,
        pagination: true,
        paginationSize: 10,
        paginationSizeSelector: [10, 25, 50, 100],
        responsiveLayout: "hide",
        movableColumns: true,
        resizableRows: true,
        minHeight: "400px",
        columnHeader: true,
        rowHeight: 30,
        headerHeight: 35,
    };

    // Merge user options with standard config
    const config = { ...standardConfig, ...options };

    // Create tabulator instance
    const table = new Tabulator(elementId, config);

    // Table hazır olduğunda container'ı oluştur
    table.on("tableBuilt", function () {
        createTabulatorContainerJQuery(elementId, table, options.buttons);
    });

    // Return tabulator instance
    return table;
}

// jQuery versiyonu için alternatif (eğer jQuery kullanmak istiyorsanız)
function createTabulatorContainerJQuery(
    elementId,
    tableInstance,
    buttons = []
) {
    elementId = elementId.replace("#", "");

    // Eğer buttons container zaten varsa, tekrar oluşturma
    if ($(`#${elementId}-tabulator-header-buttons`).length > 0) {
        return;
    }

    const div = document.createElement("div");
    div.id = `${elementId}-tabulator-header-buttons`;
    div.className = "tabulator-header-buttons mb-1 text-end";
    $(`#${elementId}`).before(div);

    var html = `
        <div class="btn-group me-2" role="group">
            <button id="${elementId}-download-csv" class="btn btn-primary btn-sm">CSV</button>
            <button id="${elementId}-download-json" class="btn btn-primary btn-sm">JSON</button>
            <button id="${elementId}-download-xlsx" class="btn btn-primary btn-sm">XLSX</button>
            <button id="${elementId}-download-pdf" class="btn btn-primary btn-sm">PDF</button>
            <button id="${elementId}-download-html" class="btn btn-primary btn-sm">HTML</button>
        </div>
    `;
    if (buttons.length > 0) {
        html += `
            <div class="btn-group me-2" role="group">
                ${buttons
                    .map(
                        (button, index) =>
                            `<button id="${elementId}-custom-btn-${index}" class="btn btn-${
                                button.icon_color || "primary"
                            } btn-sm" data-bs-toggle="popover" data-bs-trigger="hover" data-bs-placement="top" data-bs-content="${
                                button.label
                            }"><i class="${button.icon}"></i></button>`
                    )
                    .join("")}
            </div>
        `;
    }
    $(`#${elementId}-tabulator-header-buttons`).html(html);

    popoverInit();

    // Add event listeners for custom buttons
    if (buttons.length > 0) {
        buttons.forEach((button, index) => {
            $(`#${elementId}-custom-btn-${index}`).on("click", function () {
                if (typeof button.func === "function") {
                    button.func(table);
                }
            });
        });
    }

    let table = tableInstance;
    if (!table) {
        table = Tabulator.findTable(`#${elementId}`)[0];
        if (!table) {
            console.error(`Tabulator instance not found: ${elementId}`);
            return null;
        }
    }

    // jQuery event listeners
    $(`#${elementId}-download-csv`).on("click", function () {
        table.download("csv", "data.csv");
    });

    $(`#${elementId}-download-json`).on("click", function () {
        table.download("json", "data.json");
    });

    $(`#${elementId}-download-xlsx`).on("click", function () {
        table.download("xlsx", "data.xlsx", { sheetName: "My Data" });
    });

    $(`#${elementId}-download-pdf`).on("click", function () {
        table.download("pdf", "data.pdf", {
            orientation: "portrait",
            title: "Example Report",
        });
    });

    $(`#${elementId}-download-html`).on("click", function () {
        table.download("html", "data.html", { style: true });
    });

    return table;
}
// ******************* END OF SIMPLE TABULATOR WRAPPER *******************

getTheme();

function getTheme() {
    document.addEventListener("DOMContentLoaded", function () {
        var themeConfig = {
            theme: "light",
            "theme-base": "gray",
            "theme-font": "sans-serif",
            "theme-primary": "blue",
            "theme-radius": "1",
        };
        var url = new URL(window.location);
        var form = document.getElementById("offcanvasSettings");
        var resetButton = document.getElementById("reset-changes");

        // Eğer form yoksa fonksiyonu sonlandır
        if (!form) {
            return;
        }

        // Reset button kontrolü
        if (!resetButton) {
            return;
        }

        var checkItems = function () {
            for (var key in themeConfig) {
                var value =
                    window.localStorage["tabler-" + key] || themeConfig[key];
                if (!!value) {
                    var radios = form.querySelectorAll(`[name="${key}"]`);
                    if (!!radios) {
                        radios.forEach((radio) => {
                            radio.checked = radio.value === value;
                        });
                    }
                }

                if (key === "theme") {
                    $("body").removeClass("theme-light theme-dark");
                    $("body").addClass(`theme-${value}`);
                }
            }
        };
        form.addEventListener("change", function (event) {
            var target = event.target,
                name = target.name,
                value = target.value;
            for (var key in themeConfig) {
                if (name === key) {
                    document.documentElement.setAttribute(
                        "data-bs-" + key,
                        value
                    );
                    window.localStorage.setItem("tabler-" + key, value);
                    url.searchParams.set(key, value);
                }
            }
            window.history.pushState({}, "", url);
        });
        resetButton.addEventListener("click", function () {
            for (var key in themeConfig) {
                var value = themeConfig[key];
                document.documentElement.removeAttribute("data-bs-" + key);
                window.localStorage.removeItem("tabler-" + key);
                url.searchParams.delete(key);
            }
            checkItems();
            window.history.pushState({}, "", url);
        });
        checkItems();
    });
}

// --------------------- DISTANCE CALCULATION ---------------------
// // Calculate distance between two points (Haversine formula)
function calculateDistance(lat1, lon1, lat2, lon2) {
    var R = 6371; // Radius of the earth in km
    var dLat = deg2rad(lat2 - lat1);
    var dLon = deg2rad(lon2 - lon1);
    var a =
        Math.sin(dLat / 2) * Math.sin(dLat / 2) +
        Math.cos(deg2rad(lat1)) *
            Math.cos(deg2rad(lat2)) *
            Math.sin(dLon / 2) *
            Math.sin(dLon / 2);
    var c = 2 * Math.atan2(Math.sqrt(a), Math.sqrt(1 - a));
    var d = R * c; // Distance in km

    // if distance is more than 10 km, skip
    if (d > 10) {
        //console.log("distance is more than 10 km", d);
        return -0.00001;
    }
    return d;
}

function deg2rad(deg) {
    return deg * (Math.PI / 180);
}
// ******************* END OF DISTANCE CALCULATION *******************
// ---------------------  TIME RANGE CALCULATION FOR TABLE UPDATE ---------------------
function calculateTimeRange(data) {
    // calculate time range according to data length
    const timeRange = data.length * 100;
    return timeRange;
}
// ******************* END OF TIME RANGE CALCULATION *******************
// --------------------- SHOW RAPORTS DATA PROCESSING INFO  ---------------------
function showReportDataProcessingInfo(message) {
    showNotification("info", message);
}
// ******************* END OF SHOW RAPORTS DATA PROCESSING INFO *******************
