<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Log;

class GeneralSetting extends Model
{
    use HasFactory;

    protected $table = 'vts_general_settings';

    protected $fillable = [
        'mid',
        'store_id',
        'setting_key',
        'setting_value',
        'setting_type',
        'category',
        'description',
        'is_active'
    ];

    protected $casts = [
        'is_active' => 'boolean',
        'setting_value' => 'string'
    ];

    protected $appends = ['typed_value'];

    /**
     * Get setting value with proper type casting
     */
    public function getTypedValueAttribute()
    {
        if (empty($this->setting_value)) {
            return null;
        }

        switch ($this->setting_type) {
            case 'number':
                return is_numeric($this->setting_value) ? (float) $this->setting_value : 0;
            case 'boolean':
                return filter_var($this->setting_value, FILTER_VALIDATE_BOOLEAN);
            case 'json':
                return json_decode($this->setting_value, true);
            default:
                return $this->setting_value;
        }
    }

    /**
     * Set setting value with proper type handling
     */
    public function setTypedValue($value)
    {
        switch ($this->setting_type) {
            case 'number':
                $this->setting_value = is_numeric($value) ? (string) $value : '0';
                break;
            case 'boolean':
                $this->setting_value = $value ? '1' : '0';
                break;
            case 'json':
                $this->setting_value = is_array($value) ? json_encode($value) : $value;
                break;
            default:
                $this->setting_value = (string) $value;
        }
    }

    /**
     * Get a setting value by key for a specific merchant or globally
     * Falls back to default settings (mid=0, store_id=0) if not found
     */
    public static function getValue($mid = null, $key, $default = null, $storeId = null)
    {
        
        
        // First try to get user-specific setting
        $query = self::where('store_id', $storeId)
            ->where('setting_key', $key)
            ->where('is_active', true);
        if (!is_null($mid)) {
            $query->where('mid', $mid);
        }
        $setting = $query->first();
        
        if ($setting) {
            return $setting->typed_value;
        }
        
        // If not found, try to get default setting (mid=0, store_id=0)
        $defaultSetting = self::where('store_id', 0)
            ->where('mid', 0)
            ->where('setting_key', $key)
            ->where('is_active', true)
            ->first();
            
        return $defaultSetting ? $defaultSetting->typed_value : $default;
    }

    /**
     * Set a setting value by key for a specific merchant or globally
     */
    public static function setValue($mid = null, $key, $value, $type = 'string', $category = 'general', $description = null, $storeId = null)
    {
        
        //Log::info("setValue called", [$mid, $key, $value, $type, $category, $description, $storeId]);
        $where = [
            'store_id' => $storeId,
            'setting_key' => $key
        ];
        if (!is_null($mid)) {
            $where['mid'] = $mid;
        }
        $setting = self::updateOrCreate(
            $where,
            [
                'setting_value' => '',
                'setting_type' => $type,
                'category' => $category,
                'description' => $description,
                'is_active' => true
            ]
        );
        $setting->setTypedValue($value);
        $setting->save();
        return $setting;
    }

    /**
     * Get all settings for a specific merchant and category or globally
     * Merges default settings with user-specific settings (user settings override defaults)
     */
    public static function getByCategory($mid = null, $category, $storeId = null)
    {
        // 1. Get default settings for the category (mid=0, store_id=0)
        $defaultSettings = self::where('store_id', 0)
            ->where('mid', 0)
            ->where('category', $category)
            ->where('is_active', true)
            ->get()
            ->keyBy('setting_key');
        
        // 2. Get user-specific settings for the category if mid is provided
        $userSettings = collect();
        if (!is_null($mid)) {
            $userSettings = self::where('store_id', $storeId)
                ->where('mid', $mid)
                ->where('category', $category)
                ->where('is_active', true)
                ->get()
                ->keyBy('setting_key');
        }
        
        // 3. Merge settings: user settings override default settings
        $mergedSettings = collect();
        foreach ($defaultSettings as $setting) {
            if(isset($userSettings[$setting->setting_key])) {
                $mergedSettings[$setting->setting_key] = $userSettings[$setting->setting_key];
            } else {
                $mergedSettings[$setting->setting_key] = $setting;
            }
        }

        //Log::info("mergedSettings", [$mergedSettings]);
        
        return $mergedSettings;
    }

    /**
     * Get all settings for a specific merchant or globally
     * Merges default settings with user-specific settings (user settings override defaults)
     */
    public static function getAllSettings($mid = null, $storeId = null)
    {
        //Log::info("getAllSettings called", [$mid, $storeId]);
        
        // 1. Get default settings (mid=0, store_id=0)
        $defaultSettings = self::where('store_id', 0)
            ->where('mid', 0)
            ->where('is_active', true)
            ->get()
            ->keyBy('setting_key');
        
        // 2. Get user-specific settings if mid is provided
        $userSettings = collect();
        if (!is_null($mid)) {
            $userSettings = self::where('store_id', $storeId)
                ->where('mid', $mid)
                ->where('is_active', true)
                ->get()
                ->keyBy('setting_key');
        }
        
        //Log::info("userSettings", [$userSettings]);

        // 3. Merge settings: user settings override default settings
        $mergedSettings = collect();
        foreach ($defaultSettings as $setting) {
            if(isset($userSettings[$setting->setting_key])) {
                $mergedSettings[$setting->setting_key] = $userSettings[$setting->setting_key];
            } else {
                $mergedSettings[$setting->setting_key] = $setting;
            }
        }
        
        // Group by category for return
        return $mergedSettings->groupBy('category');
    }

    /**
     * Delete a setting by key for a specific merchant or globally
     */
    public static function deleteSetting($mid = null, $key, $storeId = null)
    {
        
        $query = self::where('store_id', $storeId)
            ->where('setting_key', $key);
        if (!is_null($mid)) {
            $query->where('mid', $mid);
        }
        return $query->delete();
    }

    /**
     * Get fuel price for a specific merchant or globally
     */
    public static function getFuelPrice($mid = null, $fuelType = 'diesel', $storeId = null)
    {
        $key = "fuel_price_{$fuelType}";
        return self::getValue($mid, $key, 0, $storeId);
    }

    /**
     * Set fuel price for a specific merchant or globally
     */
    public static function setFuelPrice($mid = null, $fuelType, $price, $storeId = null)
    {
        $key = "fuel_price_{$fuelType}";
        return self::setValue($mid, $key, $price, 'number', 'fuel', "Fuel price for {$fuelType}", $storeId);
    }

    /**
     * Get all fuel prices for a specific merchant or globally
     */
    public static function getAllFuelPrices($mid = null, $storeId = null)
    {
        return self::getByCategory($mid, 'fuel', $storeId);
    }

    /**
     * Get maintenance settings for a specific merchant or globally
     */
    public static function getMaintenanceSettings($mid = null, $storeId = null)
    {
        return self::getByCategory($mid, 'maintenance', $storeId);
    }

    /**
     * Get alert settings for a specific merchant or globally
     */
    public static function getAlertSettings($mid = null, $storeId = null)
    {
        return self::getByCategory($mid, 'alerts', $storeId);
    }

    /**
     * Get system settings for a specific merchant or globally
     */
    public static function getSystemSettings($mid = null, $storeId = null)
    {
        return self::getByCategory($mid, 'system', $storeId);
    }

    /**
     * Check if default settings exist, create them if they don't
     */
    public static function ensureDefaultSettingsExist()
    {
        $defaultSettingsCount = self::where('store_id', 0)
            ->where('mid', 0)
            ->where('is_active', true)
            ->count();

        if ($defaultSettingsCount === 0) {
            // Run the default settings seeder
            \Artisan::call('db:seed', ['--class' => 'DefaultSettingsSeeder', '--force' => true]);
            //Log::info('Default settings created automatically');
        }
    }
}
