<?php

namespace App\Http\Controllers;
use Illuminate\Support\Facades\DB;  // DB facade'ını burada import ediyoruz
use Illuminate\Support\Facades\Log; // Log facade'ını burada import ediyoruz
use Illuminate\Http\Request;
use App\Traits\ApiResponse;
use App\Http\Controllers\DeviceController;
use App\Http\Controllers\CategoryController;
use App\Http\Controllers\GeneralSettingController;
use App\Http\Resources\DeviceResource;
use App\Http\Resources\DeviceCoordinatesResource;
use App\Http\Resources\DeviceCategoryResource;
use App\Http\Resources\CategoryResource;

use App\Helpers\PublicHelper;
use App\Models\Device;
use DateTime;
use App\Models\DeviceLog;
use App\Models\Alarm;
use Carbon\Carbon;
use App\Helpers\Base64Helper;


class ReportController extends Controller
{

    public function visitReport()
    {
        return view('reports.visit-report');
    }

    public function dailySummary()
    {
        return view('reports.daily-summary');
    }

    /**
     * Get daily summary data for API - Simplified query
     */
    public function getDailySummaryData(Request $request)
    {
        $tokenModel = PublicHelper::DecodeJWT($request);
        $mid = $tokenModel['owenerId'];

        // Get filter parameters
        $startDate = $request->input('start_date', date('Y-m-d')) . ' 00:00:00';
        $endDate = $request->input('end_date', date('Y-m-d')) . ' 23:59:59';
        $vehicleId = $request->input('vehicle') ?? null;
        $categoryFilter = $request->input('category') ?? null;

        // Build query
        $getVehiclesLogsRequest = $request->merge([
            'did' => $vehicleId,
            'start_date' => $startDate,
            'end_date' => $endDate
        ]);
        $getCoordinates = app(DeviceController::class)->getVehiclesLogs($getVehiclesLogsRequest);
        $coordinates = $getCoordinates->getData(true)['data'] ?? [];

        $data = [
            'logs' => $coordinates,
        ];

        return response()->json([
            'success' => true,
            'data' => $data
        ]);
    }

    /**
     * Get vehicles list for filter dropdown
     */
    public function getVehiclesList(Request $request)
    {
        $tokenModel = PublicHelper::DecodeJWT($request);
        $mid = $tokenModel['owenerId'];

        $vehicles = DB::select("
            SELECT did, nm, id
            FROM vts_devices 
            WHERE mid = ? AND sts = 3
            ORDER BY nm ASC
        ", [$mid]);

        return response()->json([
            'success' => true,
            'data' => [
                'vehicles' => $vehicles
            ]
        ]);
    }

    /**
     * Get categories list for filter dropdown
     */
    public function getCategories(Request $request)
    {
        
        // categories
        $categories = app(DeviceController::class)->categories($request);

        return response()->json([
            'success' => true,
            'data' => $categories->getData(true)['data'] ?? []
        ]);
    }

    /**
     * Show parking analysis report page
     */
    public function parkingAnalysis(Request $request)
    {
        return view('reports.parking-analysis');
    }

    /**
     * API: Get parking analysis data (dummy example)
     */
    public function getParkingAnalysisData(Request $request)
    {
        $tokenModel = PublicHelper::DecodeJWT($request);
        $mid = $tokenModel['owenerId'];

        $startDate = $request->input('start_date', date('Y-m-d'));
        $endDate = $request->input('end_date', date('Y-m-d'));
        $vehicleFilter = $request->input('vehicle', '');
        
        // Optimize: Use INNER JOIN and move date filter to WHERE clause
        $query = "
            SELECT 
                d.did,
                d.nm as vehicle_name,
                d.id as vehicle_id,
                l.log_datetime as dtx,
                l.latitude as latitude,
                l.longitude as longitude,
                l.log_data_json as log_data
            FROM vts_device_logs l
            INNER JOIN vts_devices d ON d.did = l.did
            WHERE d.mid = ? 
                AND d.sts = 3
                AND l.log_datetime BETWEEN ? AND ?
                AND l.latitude IS NOT NULL
                AND l.longitude IS NOT NULL
        ";

        $params = [$mid, $startDate . ' 00:00:00', $endDate . ' 23:59:59'];

        if (!empty($vehicleFilter)) {
            $query .= " AND d.did = ?";
            $params[] = $vehicleFilter;
        }

        $query .= " ORDER BY l.log_datetime ASC LIMIT 50000";

        $rawData = DB::select($query, $params);

        $data = [
            'data' => $rawData,
        ];

        return response()->json([
            'success' => true,
            'data' => $data
        ]);
    }

    /**
     * Show route efficiency report page
     */
    public function routeEfficiency()
    {
        return view('reports.route-efficiency');
    }


    /**
     * Get raw visit data for JavaScript processing (like parking analysis)
     */
    public function getVisitData(Request $request)
    {
        try {
            $tokenModel = PublicHelper::DecodeJWT($request);
            $mid = $tokenModel['owenerId'];

            $vehicle = $request->input('vehicle');
            $startDate = $request->input('start_date', date('Y-m-d', strtotime('-7 days')));
            $endDate = $request->input('end_date', date('Y-m-d'));
            $locationType = $request->input('location_type', 'all');

            // RAW SQL Place in/out data
            // NOT USED LazyCollection

            // if ($locationType === 'all' || $locationType === 'places') {
            //     $placeQuery = "
            //         SELECT
            //             CASE 
            //                 WHEN pd.bid = 1 THEN 1
            //                 ELSE 0
            //             END AS MobilBildirim,
            //             d.`mid`,
            //             d.did,
            //             l.dt AS timestamp,
            //             DATE(l.dt) AS date_only,
            //             d.nm AS device_name,
            //             l.enl AS latitude,
            //             l.boy AS longitude,
            //             l.id AS log_id,
            //             p.id AS place_id,
            //             p.nm AS place_name,
            //             p.radius AS place_radius,
            //             CASE 
            //                 WHEN ST_Distance_Sphere(POINT(l.boy, l.enl), POINT(p.lng, p.lat)) <= p.radius THEN 1
            //                 ELSE 0
            //             END AS is_inside,
            //             CASE
            //                 WHEN ST_Distance_Sphere(POINT(l.boy, l.enl), POINT(p.lng, p.lat)) <= p.radius THEN 'entry'
            //                 ELSE 'exit'
            //             END AS visit_type,
            //             'place' AS location_type
            //         FROM
            //             vts_device_logs l
            //         INNER JOIN vts_devices d ON 
            //             d.did = l.did
            //         INNER JOIN vts_places p ON 
            //             p.`mid` = d.`mid`
            //             AND p.sts = 1
            //         INNER JOIN vts_place_devices pd ON 
            //             pd.did = l.did 
            //             AND pd.pid = p.id
            //             AND pd.`mid` = d.`mid`
            //         WHERE
            //             d.`mid` = ?
            //             AND l.enl <> 0
            //             AND l.boy <> 0
            //             AND l.dt BETWEEN ? AND ?
            //     ";

            //     $params = [$mid, $startDate . ' 00:00:00', $endDate . ' 23:59:59'];

            //     if ($vehicle) {
            //         $placeQuery .= " AND l.did = ?";
            //         $params[] = $vehicle;
            //     }

            //     $placeQuery .= " ORDER BY l.id ASC";

            //     $placeData = DB::select($placeQuery, $params);
            //     $rawData = array_merge($rawData, $placeData);
            // }

            // LazyCollection
            // Place in/out data
            if ($locationType === 'all' || $locationType === 'places') {
                
                // ✅ Query Builder kullanın - lazy() ile uyumlu
                $placeQuery = DB::table('vts_device_logs as l')
                    ->select([
                        DB::raw('CASE WHEN pd.bid = 1 THEN 1 ELSE 0 END AS mobileNotificationSended'),
                        'd.mid',
                        'd.did',
                        'l.log_datetime as timestamp',
                        DB::raw('DATE(l.log_datetime) as date_only'),
                        'd.nm as device_name',
                        'l.latitude as latitude',
                        'l.longitude as longitude',
                        'l.id as log_id',
                        'p.id as place_id',
                        'p.nm as place_name',
                        'p.radius as place_radius',
                        DB::raw('ST_Distance_Sphere(POINT(l.longitude, l.latitude), POINT(p.lng, p.lat)) as distance'),
                        DB::raw('CASE 
                            WHEN ST_Distance_Sphere(POINT(l.longitude, l.latitude), POINT(p.lng, p.lat)) <= p.radius THEN 1 
                            ELSE 0 
                        END as is_inside'),
                        DB::raw('CASE
                            WHEN ST_Distance_Sphere(POINT(l.longitude, l.latitude), POINT(p.lng, p.lat)) <= p.radius THEN "entry"
                            ELSE "exit"
                        END as visit_type'),
                        DB::raw('"place" as location_type')
                    ])
                    ->join('vts_devices as d', 'd.did', '=', 'l.did')
                    ->join('vts_places as p', function($join) {
                        $join->on('p.mid', '=', 'd.mid')
                            ->where('p.sts', '=', 1);
                    })
                    ->join('vts_place_devices as pd', function($join) {
                        $join->on('pd.did', '=', 'l.did')
                            ->on('pd.pid', '=', 'p.id')
                            ->on('pd.mid', '=', 'd.mid');
                    })
                    ->where('d.mid', $mid)
                    ->where('l.latitude', '<>', 0)
                    ->where('l.longitude', '<>', 0)
                    ->whereBetween('l.log_datetime', [$startDate . ' 00:00:00', $endDate . ' 23:59:59']);

                if ($vehicle) {
                    $placeQuery->where('l.did', $vehicle);
                }

                // ✅ Lazy ile memory-safe işlem
                $placeData = $placeQuery->orderBy('l.id')
                    ->lazy(10000);  // avarage RAM usage: 10000 records = 25MB !?             
            }

            // Get places list for map
            $placesList = [];
            if ($locationType === 'all' || $locationType === 'places') {
                $placesList = DB::table('vts_places as p')
                    ->select([
                        'p.id',
                        'p.nm as name',
                        'p.lat as latitude',
                        'p.lng as longitude',
                        'p.radius',
                        'p.sts as status',
                        'p.ord'
                    ])
                    ->where('p.mid', $mid)
                    ->where('p.sts', 1)
                    ->orderBy('p.ord')
                    ->get();
            }

            return response()->json([
                'success' => true,
                'data' => [
                    'raw_data' => $placeData ?? [],
                    'places_list' => $placesList,
                    'query_info' => [
                        'total_raw_records' => count($placeData),
                        'total_places' => count($placesList),
                        'date_range' => $startDate . ' to ' . $endDate,
                        'vehicle_filter' => $vehicle ?: __('All vehicles'),
                        'location_type' => $locationType
                    ]
                ]
            ]);

        } catch (\Exception $e) {
            \Log::error(__("Error in getVisitData") . ": " . $e->getMessage(), [
                'file' => $e->getFile(),
                'line' => $e->getLine(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => __('Error') . ': ' . $e->getMessage()
            ], 500);
        }
    }

    public function zoneViolations()
    {
        return view('reports.zone-violations');
    }

    /**
     * Get raw region data for JavaScript processing
     */
    public function getRegionData(Request $request)
    {
        try {
            $tokenModel = PublicHelper::DecodeJWT($request);
            $mid = $tokenModel['owenerId'];

            $vehicle = $request->input('vehicle');
            $startDate = $request->input('start_date', date('Y-m-d', strtotime('-7 days')));
            $endDate = $request->input('end_date', date('Y-m-d'));
            //toll analysis
            $vehicleHasPrice = $request->input('vehicle_has_price', 0);
            $regionStatus = $request->input('region_status', 0);

            $vehicleHasPriceQuery = "";
            if($vehicleHasPrice == 1){
                $vehicleHasPriceQuery = " AND pd.price > 0";
            }

            $regionStatusQuery = "";
            if($regionStatus > 0){
                $regionStatusQuery = " AND pd.zone_type = $regionStatus";
            }
            
            $rawData = [];

            // Region in/out data
                $regionQuery = "
                    SELECT
                        CASE 
                            WHEN pd.zone_type = 1 THEN 'no_entry'
                            WHEN pd.zone_type = 2 THEN 'no_exit'
                        END AS region_status,
                        CASE 
                            WHEN ST_Contains(p.poly, l.coord) THEN 1
                            ELSE 0
                        END AS poly_contains,
                        CASE
                            WHEN pd.zone_type = 1 AND ST_Contains(p.poly, l.coord) THEN 'entry'
                            WHEN pd.zone_type = 1 AND NOT ST_Contains(p.poly, l.coord) THEN 'exit'
                            WHEN pd.zone_type = 2 AND NOT ST_Contains(p.poly, l.coord) THEN 'exit'
                            WHEN pd.zone_type = 2 AND ST_Contains(p.poly, l.coord) THEN 'entry'
                        ELSE '-'
                        END AS region_control,
                        CASE 
                            WHEN pd.bid = 1 THEN 1
                            ELSE 0
                        END AS mobile_notification,
                        pd.zone_type AS region_control_int,
                        pd.price AS region_price,
                        d.`mid`,
                        d.did,
                        l.log_datetime AS event_time,
                        DATE(l.log_datetime) AS event_date,
                        d.nm AS device_name,
                        l.latitude AS latitude,
                        l.longitude AS longitude,
                        l.id AS log_id,
                        p.id AS region_id,
                        p.nm AS region_name,
                        pd.zone_type AS poly_status,
                        l.log_data_json as log_details,
                        'region' AS location_type
                    FROM
                        vts_device_logs l
                    INNER JOIN vts_devices d ON 
                        d.did = l.did
                    INNER JOIN vts_regions p ON 
                        p.`mid` = d.`mid`
                    INNER JOIN vts_region_devices pd ON 
                        pd.did = l.did 
                        AND pd.pid = p.id
                        AND pd.`mid` = d.`mid`
                        AND pd.zone_type <> 3
                        $vehicleHasPriceQuery
                        $regionStatusQuery
                    WHERE
                        d.`mid` = ?
                        AND l.latitude <> 0
                        AND l.longitude <> 0
                        AND l.log_datetime BETWEEN ? AND ?
                ";

                $params = [$mid, $startDate . ' 00:00:00', $endDate . ' 23:59:59'];

                if ($vehicle) {
                    $regionQuery .= " AND l.did = ?";
                    $params[] = $vehicle;
                }

                $regionQuery .= " ORDER BY l.id ASC";

                $regionData = DB::select($regionQuery, $params);
                $rawData = array_merge($rawData, $regionData);
            

            // Get regions list for map
            $regionsList = [];
                $regionsList = DB::table('vts_regions as poly')
                    ->select([
                        'poly.id',
                        'poly.nm as name',
                        'poly.ply as polygon_json',
                        'poly.active as status',
                        'poly.ord'
                    ])
                    ->where('poly.mid', $mid)
                    ->where('poly.active', 1)
                    ->orderBy('poly.ord')
                    ->get();

            return response()->json([
                'success' => true,
                'data' => [
                    'raw_data' => $rawData,
                    'regions_list' => $regionsList,
                    'query_info' => [
                        'total_raw_records' => count($rawData),
                        'total_regions' => count($regionsList),
                        'date_range' => $startDate . ' to ' . $endDate,
                        'vehicle_filter' => $vehicle ?: __('All vehicles')
                    ]
                ]
            ]);

        } catch (\Exception $e) {
            \Log::error(__("Error in getRegionData") . ": " . $e->getMessage(), [
                'file' => $e->getFile(),
                'line' => $e->getLine(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => __('Error') . ': ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get user ID from JWT token in the request cookies
     */
    private function getUserIdFromToken(Request $request)
    {
        $tokenModel = PublicHelper::DecodeJWT($request);
        return $tokenModel['userId'] ?? null;
    }

    /**
     * Generate after-hours usage report - returns raw data for JS processing
     */
    public function afterHoursData(Request $request)
    {
        try {
            $mid = $this->getUserIdFromToken($request);
            if (!$mid) {
                return response()->json([
                    'success' => false,
                    'message' => 'Unauthorized'
                ], 401);
            }

            $data = $request->has('data') ?
                Base64Helper::decode($request->input('data')) :
                $request->all();

            $deviceId = $data['device_id'] ?? null;
            $startDate = $data['start_date'] ?? Carbon::now()->startOfMonth();
            $endDate = $data['end_date'] ?? Carbon::now()->endOfMonth();

            if (!$deviceId) {
                return response()->json([
                    'success' => false,
                    'message' => 'Device ID is required'
                ], 400);
            }

            // Get after-hours settings for this device
            $afterHoursAlarm = Alarm::where('mid', $mid)
                ->where('did', $deviceId)
                ->where('aid', 307) // After Hours alarm form ID
                ->first();

            if (!$afterHoursAlarm || !$afterHoursAlarm->did) {
                return response()->json([
                    'success' => false,
                    'message' => 'After-hours monitoring is not enabled for this device'
                ], 400);
            }

            // Get device logs for the date range
            $deviceLogs = DeviceLog::where('did', $deviceId)
                ->whereBetween('log_datetime', [$startDate, $endDate])
                ->orderBy('log_datetime')
                ->lazy(1000)
                ->map(function($log) {
                    // log_data_json is already cast as array in the model
                    $lg = $log->log_data_json ?? null;
                    
                    return [
                        'id' => $log->id,
                        'timestamp' => $log->log_datetime->format('Y-m-d H:i:s'),
                        'latitude' => $log->latitude,
                        'longitude' => $log->longitude,
                        'speed' => $lg['hiz'] ?? null,
                        'direction' => null, // dm column was deleted
                        'log_data' => $lg
                    ];
                });

            // Get device information
            $device = Device::where('did', $deviceId)->first();

            return response()->json([
                'success' => true,
                'data' => [
                    'device' => $device,
                    'settings' => $afterHoursAlarm->settings_json,
                    'report_period' => [
                        'start_date' => $startDate,
                        'end_date' => $endDate
                    ],
                    'raw_logs' => $deviceLogs
                ]
            ]);

        } catch (\Exception $e) {
            \Log::error('Error in afterHoursReport: ' . $e->getMessage(), [
                'file' => $e->getFile(),
                'line' => $e->getLine(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Error generating report: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * page speed violations
     */
    public function getSpeedViolations(Request $request)
    {
        return view('reports.speed-violations');
    }   

    /**
     * Get speed violations data
     */
    public function getSpeedViolationsData(Request $request)
    {
        $tokenModel = PublicHelper::DecodeJWT($request);
        $mid = $tokenModel['owenerId'];

        $vehicleId = $request->input('device_id');
        $startDate = $request->input('start_date', date('Y-m-d', strtotime('-7 days'))). ' 00:00:00';
        $endDate = $request->input('end_date', date('Y-m-d')) . ' 23:59:59';

        $getVehiclesLogsRequest = $request->merge([
            'did' => $vehicleId,
            'start_date' => $startDate,
            'end_date' => $endDate
        ]);
        $getCoordinates = app(DeviceController::class)->getVehiclesLogs($getVehiclesLogsRequest);
        $coordinates = $getCoordinates->getData(true)['data'] ?? [];

        return response()->json([
            'success' => true,
            'data' => $coordinates,
        ]);
    }

    /**
     * harsh-driving
     */
    public function harshDriving()
    {
        return view('reports.harsh-driving');
    }

    /**
     * Get harsh driving data
     */ 
    public function getHarshDrivingData(Request $request)
    {
        $tokenModel = PublicHelper::DecodeJWT($request);
        $mid = $tokenModel['owenerId'];

        $vehicleId = $request->input('device_id');
        $startDate = $request->input('start_date', date('Y-m-d', strtotime('-7 days'))) . ' 00:00:00';
        $endDate = $request->input('end_date', date('Y-m-d')) . ' 23:59:59';

        // get coordinates
        $getVehiclesLogsRequest = $request->merge([
            'did' => $vehicleId,
            'start_date' => $startDate,
            'end_date' => $endDate
        ]);
        $getCoordinates = app(DeviceController::class)->getVehiclesLogs($getVehiclesLogsRequest);
        $coordinates = $getCoordinates->getData(true)['data'] ?? [];
           
        return response()->json([
            'success' => true,
            'data' => $coordinates,
        ]);
    }

    /**
     * fuel consumption
     */
    public function fuelConsumption()
    {
        return view('reports.fuel-consumption');
    }

    /**
     * Get fuel consumption data
     */
    public function getFuelConsumptionData(Request $request)
    {
        $tokenModel = PublicHelper::DecodeJWT($request);
        $mid = $tokenModel['owenerId'];

        $vehicleId = $request->input('device_id');
        $startDate = $request->input('start_date', date('Y-m-d', strtotime('-7 days'))). ' 00:00:00';
        $endDate = $request->input('end_date', date('Y-m-d')) . ' 23:59:59';
        
        // get coordinates
        $getVehiclesLogsRequest = $request->merge([
            'did' => $vehicleId,
            'start_date' => $startDate,
            'end_date' => $endDate
        ]);
        $getCoordinates = app(DeviceController::class)->getVehiclesLogs($getVehiclesLogsRequest);
        $coordinates = $getCoordinates->getData(true)['data'] ?? [];


        $fuelPrices = app(GeneralSettingController::class)->getFuelPrices($request);
        $fuelPrices = $fuelPrices->getData(true)['data'] ?? [];

        return response()->json([
            'success' => true,
            'data' => [
                'device_logs' => $coordinates,
                'fuel_prices' => $fuelPrices
            ]
        ]);
    }

    /**
     * toll analysis
     */
    public function tollAnalysis()
    {
        return view('reports.toll-analysis');
    }
    
    /**
     * Get toll analysis data
     */
    public function getTollAnalysisData(Request $request)
    {
        try {
            $tokenModel = PublicHelper::DecodeJWT($request);
            $mid = $tokenModel['owenerId'];

            $vehicleId = $request->input('device_id');
            $startDate = $request->input('start_date', date('Y-m-d', strtotime('-7 days')));
            $endDate = $request->input('end_date', date('Y-m-d'));

            // Get device logs from getRegionData
            $requestGetRegionData = $request->merge([
                'start_date' => $startDate, 
                'end_date' => $endDate, 
                'vehicle' => $vehicleId,
                'vehicle_has_price' => 1,
                //'region_status' => 1
            ]);
            $regionDataResponse = $this->getRegionData($requestGetRegionData);
            $regionData = $regionDataResponse->getData(true)['data']['raw_data'] ?? [];
            
            return response()->json([
                'success' => true,
                'data' => [
                    'device_logs' => $regionData
                ]
            ]);

        } catch (\Exception $e) {
            Log::error('Error in getTollAnalysisData: ' . $e->getMessage(), [
                'file' => $e->getFile(),
                'line' => $e->getLine(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Error generating toll analysis: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * working-hours page
     */
    public function workingHours()
    {
        return view('reports.working-hours');
    }

    /**
     * Get working hours utilization data - returns raw data for JS processing
     */
    public function getWorkingHoursData(Request $request)
    {
        try {
            $mid = $this->getUserIdFromToken($request);
            if (!$mid) {
                return response()->json([
                    'success' => false,
                    'message' => 'Unauthorized'
                ], 401);
            }

            $data = $request->has('data') ?
                Base64Helper::decode($request->input('data')) :
                $request->all();

            $deviceId = $data['device_id'] ?? null;
            $startDate = $data['start_date'] ?? Carbon::now()->startOfMonth();
            $endDate = $data['end_date'] ?? Carbon::now()->endOfMonth();

            if (!$deviceId) {
                return response()->json([
                    'success' => false,
                    'message' => 'Device ID is required'
                ], 400);
            }

            // Get business hours settings for this device (same alarm as after-hours)
            $businessHoursAlarm = Alarm::where('mid', $mid)
                ->where('did', $deviceId)
                ->where('aid', 307) // Business Hours alarm form ID (same as after-hours)
                ->first();

            if (!$businessHoursAlarm || !$businessHoursAlarm->did) {
                return response()->json([
                    'success' => false,
                    'message' => 'Business hours monitoring is not enabled for this device'
                ], 400);
            }

            // Get device logs for the date range
            $deviceLogs = DeviceLog::where('did', $deviceId)
                ->whereBetween('log_datetime', [$startDate . ' 00:00:00', $endDate . ' 23:59:59'])
                ->orderBy('log_datetime')
                ->get();

            // Get device information
            $device = Device::where('did', $deviceId)->first();

            return response()->json([
                'success' => true,
                'data' => [
                    'device' => $device,
                    'settings' => $businessHoursAlarm->settings_json,
                    'report_period' => [
                        'start_date' => $startDate,
                        'end_date' => $endDate
                    ],
                    'raw_logs' => $deviceLogs->map(function($log) use ($device) {
                        // log_data_json is already cast as array in the model
                        $lg = $log->log_data_json ?? null;
                        return [
                            'id' => $log->id,
                            'device_id' => $log->did,
                            'vehicle_name' => $device ? $device->nm : null,
                            'timestamp' => $log->log_datetime->format('Y-m-d H:i:s'),
                            'latitude' => $log->latitude,
                            'longitude' => $log->longitude,
                            'speed' => $lg['hiz'] ?? 0,
                            'direction' => null, // dm column was deleted
                            'log_data' => $lg
                        ];
                    })
                ]
            ]);

        } catch (\Exception $e) {
            \Log::error('Error in getWorkingHoursData: ' . $e->getMessage(), [
                'file' => $e->getFile(),
                'line' => $e->getLine(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Error generating working hours report: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * inspection-schedule page
     */
    public function inspectionSchedule()
    {
        return view('reports.inspection-schedule');
    }
    
    /**
     * Get inspection schedule data
     */
    public function getInspectionScheduleData(Request $request)
    {
        $tokenModel = PublicHelper::DecodeJWT($request);
        $mid = $tokenModel['owenerId'];

        $requestGetVehiclesListWithAlarm = $request->merge([
            'alarm_id' => 301,
        ]);
        $vehicles = app(DeviceController::class)->getVehiclesListWithAlarm($requestGetVehiclesListWithAlarm);
        $vehicles = $vehicles->getData(true)['data']['vehicles'] ?? [];

        return response()->json([
            'success' => true,
            'data' => [
                'vehicles' => $vehicles
            ]
        ]);
    }

    /**
     * mileage-tracking page
     */
    public function mileageTracking()
    {
        return view('reports.mileage-tracking');
    }
    
    /**
     * Get mileage tracking data
     */
    public function getMileageTrackingData(Request $request)
    {
        try {
            $tokenModel = PublicHelper::DecodeJWT($request);
            $mid = $tokenModel['owenerId'];

            // Build the query based on your SQL
            $query = "
                SELECT
                    d.nm as vehicle_name,
                    d.did as vehicle_id,
                    d.id as device_id,
                    JSON_UNQUOTE(JSON_EXTRACT(a.settings_json, '$.maintenance_km_date')) AS maintenance_km_date,
                    JSON_EXTRACT(a.settings_json, '$.maintenance_km') * 1 AS maintenance_km_interval,
                    
                    (SELECT CAST(JSON_EXTRACT(log_data_json, '$.mes') AS DECIMAL(10,2)) 
                     FROM vts_device_logs 
                     WHERE did = d.did 
                     AND DATE(log_datetime) >= JSON_UNQUOTE(JSON_EXTRACT(a.settings_json, '$.maintenance_km_date'))
                     ORDER BY log_datetime ASC
                     LIMIT 1) AS km_start,
                    
                    (SELECT CAST(JSON_EXTRACT(log_data_json, '$.mes') AS DECIMAL(10,2)) 
                     FROM vts_device_logs 
                     WHERE did = d.did 
                     ORDER BY id DESC 
                     LIMIT 1) AS km_end

                FROM vts_devices d
                LEFT JOIN vts_device_alarms a ON a.did = d.did AND a.aid = 306
                WHERE d.mid = ? AND d.sts = 3
                ORDER BY d.nm ASC
            ";

            $vehicles = DB::select($query, [$mid]);

            return response()->json([
                'success' => true,
                'data' => [
                    'vehicles' => $vehicles
                ]
            ]);

        } catch (\Exception $e) {
            \Log::error('Error in getMileageTrackingData: ' . $e->getMessage(), [
                'file' => $e->getFile(),
                'line' => $e->getLine(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Error loading mileage tracking data: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * vehicle-health page
     */
    public function vehicleHealth()
    {
        return view('reports.vehicle-health');
    }
    
    /**
     * Get vehicle health raw data for JavaScript processing
     */
    public function getVehicleHealthData(Request $request)
    {
        try {
            $tokenModel = PublicHelper::DecodeJWT($request);
            $mid = $tokenModel['owenerId'];

            $vehicleId = $request->input('vehicle_id');
            $startDate = $request->input('start_date', date('Y-m-d', strtotime('-1 month')));
            $endDate = $request->input('end_date', date('Y-m-d'));

            // Get vehicles
            $vehiclesQuery = Device::where('mid', $mid)->where('sts', 3);
            if ($vehicleId) {
                $vehiclesQuery->where('did', $vehicleId);
            }
            $vehicles = $vehiclesQuery->get();

            // Get device logs for the date range
            $deviceLogsQuery = DeviceLog::whereBetween('log_datetime', [$startDate . ' 00:00:00', $endDate . ' 23:59:59']);
            if ($vehicleId) {
                $deviceLogsQuery->where('did', $vehicleId);
            } else {
                $vehicleIds = $vehicles->pluck('did')->toArray();
                if (!empty($vehicleIds)) {
                    $deviceLogsQuery->whereIn('did', $vehicleIds);
                }
            }
            $deviceLogs = $deviceLogsQuery->orderBy('log_datetime')->get();

            // Get vehicles with alarm settings (inspection data)
            $requestGetVehiclesListWithAlarm = $request->merge([
                'alarm_id' => "301", //inspection
            ]);
            $vehiclesAlarmsResponse = app(DeviceController::class)->getVehiclesListWithAlarm($requestGetVehiclesListWithAlarm);
            $vehiclesAlarms = $vehiclesAlarmsResponse->getData(true)['data']['vehicles'] ?? [];

            //getMileageTrackingData 
            $alarmMileageTracking = $this->getMileageTrackingData($request);
            $alarmMileageTracking = $alarmMileageTracking->getData(true)['data']['vehicles'] ?? [];

            return response()->json([
                'success' => true,
                'data' => [
                    'vehicles' => DeviceResource::collection(collect($vehicles), $tokenModel),
                    'device_logs' => $deviceLogs->map(function($log) {
                        // log_data_json is already cast as array in the model
                        $lg = $log->log_data_json ?? null;
                        return [
                            'did' => $log->did,
                            'dtx' => $log->log_datetime ? $log->log_datetime->format('Y-m-d H:i:s') : null,
                            'log_datetime' => $log->log_datetime ? $log->log_datetime->format('Y-m-d H:i:s') : null,
                            'speed' => $lg['hiz'] ?? 0,

                        ];
                    }),
                    'alarms_inspection' => $vehiclesAlarms,
                    'alarms_mileage' => $alarmMileageTracking,
                    'query_info' => [
                        'total_vehicles' => $vehicles->count(),
                        'total_logs' => $deviceLogs->count(),
                        'date_range' => $startDate . ' to ' . $endDate,
                        'vehicle_filter' => $vehicleId ?: 'All vehicles'
                    ]
                ]
            ]);

        } catch (\Exception $e) {
            \Log::error('Error in getVehicleHealthData: ' . $e->getMessage(), [
                'file' => $e->getFile(),
                'line' => $e->getLine(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Error loading vehicle health data: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get vehicle health details for modal
     */
    public function getVehicleHealthDetails(Request $request)
    {
        try {
            $tokenModel = PublicHelper::DecodeJWT($request);
            $mid = $tokenModel['owenerId'];

            $vehicleId = $request->input('vehicle_id');
            
            if (!$vehicleId) {
                return response()->json([
                    'success' => false,
                    'message' => 'Vehicle ID is required'
                ], 400);
            }

            $vehicle = Device::where('mid', $mid)
                ->where('did', $vehicleId)
                ->where('sts', 3)
                ->first();

            if (!$vehicle) {
                return response()->json([
                    'success' => false,
                    'message' => 'Vehicle not found'
                ], 404);
            }

            // Get device logs for this vehicle
            $deviceLogs = DeviceLog::where('did', $vehicleId)
                ->whereBetween('log_datetime', [Carbon::now()->subDays(30)->format('Y-m-d') . ' 00:00:00', Carbon::now()->format('Y-m-d') . ' 23:59:59'])
                ->orderBy('log_datetime')
                ->get();

            // Get alarms for this vehicle
            $alarms = Alarm::where('mid', $mid)
                ->where('did', $vehicleId)
                ->get();

            return response()->json([
                'success' => true,
                'data' => [
                    'vehicle' => [
                        'did' => $vehicle->did,
                        'nm' => $vehicle->nm,
                        'id' => $vehicle->id,
                        'sts' => $vehicle->sts
                    ],
                    'device_logs' => $deviceLogs->map(function($log) {
                        // log_data_json is already cast as array in the model
                        $lg = $log->log_data_json ?? null;
                        return [
                            'id' => $log->id,
                            'log_datetime' => $log->log_datetime->format('Y-m-d H:i:s'),
                            'latitude' => $log->latitude,
                            'longitude' => $log->longitude,
                            'speed' => $lg['hiz'] ?? 0,
                            'log_data' => $lg
                        ];
                    }),
                    'alarms' => $alarms->map(function($alarm) {
                        return [
                            'id' => $alarm->id,
                            'aid' => $alarm->aid,
                            'settings_json' => $alarm->settings_json
                        ];
                    })
                ]
            ]);

        } catch (\Exception $e) {
            \Log::error('Error in getVehicleHealthDetails: ' . $e->getMessage(), [
                'file' => $e->getFile(),
                'line' => $e->getLine(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Error loading vehicle details: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * driver-scoring page
     */
    public function driverScoring()
    {
        return view('reports.driver-scoring');
    }

    /**
     * Get driver scoring data
     */
    public function getDriverScoringData(Request $request)
    {
        $tokenModel = PublicHelper::DecodeJWT($request);
        $mid = $tokenModel['owenerId'];

        $vehicleId = $request->input('device_id');
        $startDate = $request->input('start_date', date('Y-m-d', strtotime('-7 days')));
        $endDate = $request->input('end_date', date('Y-m-d'));

        
        // ------------------------- Device Settings -------------------------
        $requestGetVehiclesListWithAlarm = $request->merge([
            'alarm_id' => "304,305,307", // 304 speed limit, 305 harsh driving, 307 working hours
        ]);
        $deviceAlarmSettings = app(DeviceController::class)->getVehiclesListWithAlarm($requestGetVehiclesListWithAlarm);
        $deviceAlarmSettings = $deviceAlarmSettings->getData(true)['data']['vehicles'] ?? [];

        // ------------------------- END Device Settings -------------------------

        // ------------------------- Zone Settings -------------------------
        
        $zoneViolations = $this->getRegionData($request);
        $zoneViolations = $zoneViolations->getData(true)['data'] ?? [];
        // ------------------------- END Zone Settings -------------------------

        return response()->json([
            'success' => true,
            'data' => [
                'device_alarm_settings' => $deviceAlarmSettings,
                'zone_data' => $zoneViolations['raw_data'] ?? [], //logs
                'zone_list' => $zoneViolations['regions_list'] ?? [],
                'zone_query_info' => $zoneViolations['query_info'] ?? []
            ]
        ]);
    }
}