<?php

namespace App\Helpers;

use App\Models\GeneralSetting;

class SystemSettingsHelper
{
    /**
     * Get system settings from JWT token or database
     */
    public static function getSystemSettings($mid = null, $storeId = null)
    {
        // First try to get from JWT token
        if (request()->hasCookie(config('jwt.public_key'))) {
            try {
                $tokenModel = PublicHelper::DecodeJWT(request());
                
                return [
                    'locale' => $tokenModel['locale'] ?? 'en',
                    'timezone' => $tokenModel['timezone'] ?? 'Europe/Istanbul',
                    'dateFormat' => $tokenModel['dateFormat'] ?? 'd.m.Y',
                    'timeFormat' => $tokenModel['timeFormat'] ?? 'H:i',
                    'autoBackupEnabled' => $tokenModel['autoBackupEnabled'] ?? true,
                    'backupFrequency' => $tokenModel['backupFrequency'] ?? 'daily',
                ];
            } catch (\Exception $e) {
                // Fall back to database
            }
        }
        
        // Get from database if JWT token not available
        $systemSettings = GeneralSetting::getSystemSettings($mid, $storeId);
        
        return [
            'locale' => $systemSettings->get('language')?->typed_value ?? 'en',
            'timezone' => $systemSettings->get('timezone')?->typed_value ?? 'Europe/Istanbul',
            'dateFormat' => $systemSettings->get('date_format')?->typed_value ?? 'd.m.Y',
            'timeFormat' => $systemSettings->get('time_format')?->typed_value ?? 'H:i',
            'autoBackupEnabled' => $systemSettings->get('auto_backup_enabled')?->typed_value ?? true,
            'backupFrequency' => $systemSettings->get('backup_frequency')?->typed_value ?? 'daily',
        ];
    }

    /**
     * Get locale from JWT token or database
     */
    public static function getLocale($mid = null, $storeId = null)
    {
        $settings = self::getSystemSettings($mid, $storeId);
        return $settings['locale'];
    }

    /**
     * Get timezone from JWT token or database
     */
    public static function getTimezone($mid = null, $storeId = null)
    {
        $settings = self::getSystemSettings($mid, $storeId);
        return $settings['timezone'];
    }

    /**
     * Get date format from JWT token or database
     */
    public static function getDateFormat($mid = null, $storeId = null)
    {
        $settings = self::getSystemSettings($mid, $storeId);
        return $settings['dateFormat'];
    }

    /**
     * Get time format from JWT token or database
     */
    public static function getTimeFormat($mid = null, $storeId = null)
    {
        $settings = self::getSystemSettings($mid, $storeId);
        return $settings['timeFormat'];
    }

    /**
     * Format date according to user's settings
     */
    public static function formatDate($date, $mid = null, $storeId = null)
    {
        if (!$date) return null;

        $dateFormat = self::getDateFormat($mid, $storeId);
        
        $timezone = self::getTimezone($mid, $storeId);
        
        try {
            $dateTime = new \DateTime($date);
            $dateTime->setTimezone(new \DateTimeZone($timezone));
            return $dateTime->format($dateFormat);
        } catch (\Exception $e) {
            return $date;
        }
    }

    /**
     * Format time according to user's settings
     */
    public static function formatTime($time, $mid = null, $storeId = null)
    {
        if (!$time) return null;
        
        $timeFormat = self::getTimeFormat($mid, $storeId);
        $timezone = self::getTimezone($mid, $storeId);
        
        try {
            $dateTime = new \DateTime($time);
            $dateTime->setTimezone(new \DateTimeZone($timezone));
            return $dateTime->format($timeFormat);
        } catch (\Exception $e) {
            return $time;
        }
    }

    /**
     * Format datetime according to user's settings
     */
    public static function formatDateTime($datetime, $mid = null, $storeId = null)
    {
        if (!$datetime) return null;
        
        $dateFormat = self::getDateFormat($mid, $storeId);
        $timeFormatString = self::getTimeFormat($mid, $storeId);
        $timeFormatMap = config('app.timeFormatMap', []);
        $phpTimeFormat = $timeFormatMap[$timeFormatString] ?? 'H:i:s'; // default fallback

        Log::info(["phpTimeFormat" => $phpTimeFormat]);

        $timezone = self::getTimezone($mid, $storeId);
        
        try {
            $dateTime = new \DateTime($datetime);
            $dateTime->setTimezone(new \DateTimeZone($timezone));
            return $dateTime->format($dateFormat . ' ' . $phpTimeFormat);
        } catch (\Exception $e) {
            return $datetime;
        }
    }
} 